import React, { useState, useEffect } from 'react';
import {
  Layout,
  CoreText,
  TextType,
  FontSize,
  Tabs,
  JustifyContent,
  Display,
  Button,
  ButtonType,
  Position,
  DropDownMenuItem,
  DropDownMenu,
  SVGAsset,
  BalloonDirection,
  StyledLayout,
} from 'twitch-core-ui';
import { OrganizationForm } from '~features/organizations/components/organization-form';
import { OrganizationInformation } from '~pages/organization-detail/models/organization-information';
import { useRouteMatch, useLocation, Switch, Route } from 'react-router';
import { OrganizationUsersPage } from '~pages/organization-users';
import { Company } from '~core/clients/rbac/code.justin.tv.devrel';
import { OrganizationGamesPage } from '~pages/organization-games';
import { ClickOutDetector } from '~core/components/click-out-detector';
import { OrganizationPendingGamePage } from '~pages/organization-pending-games';
import { OrganizationExtensionsPage } from '~pages/organization-extensions';
import { OrganizationDevelopersPage } from '~pages/organization-developers/organization-developers-page';
import { OrganizationE2Page } from '~pages/organization-e2/organization-e2';

export interface PublicProps {
  company: Company;
  onCompanyUpdate: (values: OrganizationInformation) => void;
  onActionHistory: () => void;
  onAddUser: () => void;
  onAddShadowAccount: () => void;
  onAddGame: () => void;
  onTransferExtension: () => void;
  onTransferOwnership: () => void;
  onAllowlistE2: () => void;
}

type Props = PublicProps;

export const OrganizationDetailPresentation: React.FC<Props> = (props) => {
  const [currentTab, setCurrentTab] = useState(0);
  const [showToolMenu, setShowToolMenu] = useState(false);

  const match = useRouteMatch();
  const location = useLocation();

  useEffect(() => {
    const pathParts = location.pathname.split('/');
    const lastPath = pathParts[pathParts.length - 1];

    switch (lastPath) {
      case 'games':
        setCurrentTab(1);
        break;
      case 'e2':
        setCurrentTab(2);
        break;
      case 'pending-games':
        setCurrentTab(3);
        break;
      case 'pending-developers':
        setCurrentTab(4);
        break;
      default:
        setCurrentTab(0);
        break;
    }
  }, [location.pathname]);

  return (
    <>
      <Layout margin={2}>
        <Layout
          display={Display.Flex}
          justifyContent={JustifyContent.Between}
          margin={{ bottom: 4 }}
        >
          <Layout>
            <CoreText type={TextType.H1}>{props.company.companyName}</CoreText>
            <CoreText>First all the details for the organization below</CoreText>
          </Layout>
          <ClickOutDetector onClickOut={() => setShowToolMenu(false)}>
            <StyledLayout position={Position.Relative}>
              <Button
                type={ButtonType.Primary}
                dropdown
                onClick={() => setShowToolMenu((show) => !show)}
              >
                Management Tools
              </Button>
              <DropDownMenu show={showToolMenu} direction={BalloonDirection.BottomRight}>
                <DropDownMenuItem figure={{ icon: SVGAsset.AddUser }} onClick={props.onAddUser}>
                  Add User
                </DropDownMenuItem>
                <DropDownMenuItem
                  figure={{ icon: SVGAsset.Account }}
                  onClick={props.onAddShadowAccount}
                >
                  Add Shadow Account
                </DropDownMenuItem>
                <DropDownMenuItem figure={{ icon: SVGAsset.NavGames }} onClick={props.onAddGame}>
                  Add Game
                </DropDownMenuItem>
                <DropDownMenuItem
                  figure={{ icon: SVGAsset.Extensions }}
                  onClick={props.onTransferExtension}
                >
                  Transfer Extension
                </DropDownMenuItem>
                <DropDownMenuItem
                  figure={{ icon: SVGAsset.AddUser }}
                  onClick={props.onTransferOwnership}
                >
                  Transfer Ownership
                </DropDownMenuItem>
                <DropDownMenuItem figure={{ icon: SVGAsset.OP }} onClick={props.onAllowlistE2}>
                  Allowlist E2 Client
                </DropDownMenuItem>
                <DropDownMenuItem
                  figure={{ icon: SVGAsset.ViewerList }}
                  onClick={props.onActionHistory}
                >
                  Activity Log
                </DropDownMenuItem>
              </DropDownMenu>
            </StyledLayout>
          </ClickOutDetector>
        </Layout>

        <Layout margin={{ bottom: 4 }}>
          <OrganizationForm
            showContractOptions
            organization={props.company}
            onSubmit={props.onCompanyUpdate}
          />
        </Layout>
        <Layout>
          <Layout margin={{ bottom: 4 }}>
            <CoreText type={TextType.H2} fontSize={FontSize.Size3}>
              Organization Management
            </CoreText>
            <CoreText>Manage the organizations users and games using the tabs below</CoreText>
          </Layout>
          <Layout>
            <Layout margin={{ bottom: 2 }}>
              <Tabs
                justifyContent={JustifyContent.Start}
                tabs={[
                  {
                    linkTo: `${match.url}`,
                    label: 'Users',
                  },
                  {
                    linkTo: `${match.url}/games`,
                    label: 'Games',
                  },
                  {
                    linkTo: `${match.url}/e2`,
                    label: 'E2 Client Allowlist',
                  },
                  {
                    linkTo: `${match.url}/pending-games`,
                    label: 'Pending Games',
                  },
                  {
                    linkTo: `${match.url}/pending-developers`,
                    label: `Pending Developers`,
                  },
                  {
                    linkTo: `${match.url}/extensions`,
                    label: 'Extensions',
                  },
                ]}
                activeTabIndex={currentTab}
              />
            </Layout>

            <Switch>
              <Route component={OrganizationUsersPage} path={`${match.path}`} exact />
              <Route component={OrganizationGamesPage} path={`${match.path}/games`} />
              <Route component={OrganizationE2Page} path={`${match.path}/e2`} />
              <Route component={OrganizationPendingGamePage} path={`${match.path}/pending-games`} />
              <Route
                component={OrganizationDevelopersPage}
                path={`${match.path}/pending-developers`}
              />
              <Route component={OrganizationExtensionsPage} path={`${match.path}/extensions`} />
            </Switch>
          </Layout>
        </Layout>
      </Layout>
    </>
  );
};
