import React, { useState, useEffect, useCallback } from 'react';
import { usePagination } from '~features/hooks/use-pagination';
import { DebounceTimer } from '~core/utils/debounce-timer';
import {
  Company,
  ListCompaniesRequest,
  SearchCompaniesRequest,
} from '~core/clients/rbac/code.justin.tv.devrel';
import { RBACClient } from '~core/vienna';
import { EntityList } from '~features/entity-list/entity-list';
import { OrganizationListItem } from '~features/organization-list/list-item';
import { CheckBox } from 'twitch-core-ui';
import { GameAutocomplete } from '~features/game-autocomplete';
import { Game } from '~features/game-autocomplete/models/game';

const ORGANIZATION_LIMIT = 25;

const debounceTimer = new DebounceTimer(200);

export const OrganizationListPage: React.FC = () => {
  const { page, nextPage, previousPage, goToPage } = usePagination();
  const [gamesPending, setGamesPending] = useState(false);
  const [loading, setLoading] = useState(false);
  const [developersPending, setDevelopersPending] = useState(false);
  const [companies, setCompanies] = useState<Company[]>([]);
  const [total, setTotal] = useState(0);
  const [search, setSearch] = useState('');

  const offset = (page - 1) * 25;

  async function getCompanies() {
    setLoading(true);

    const companyResponse = await RBACClient.listCompanies(
      new ListCompaniesRequest({
        limit: ORGANIZATION_LIMIT,
        offset,
        hasDevelopersPending: developersPending,
        hasGamesPending: gamesPending,
      }),
    );

    setCompanies(companyResponse.companies);
    setTotal(Math.ceil(companyResponse.Total / ORGANIZATION_LIMIT));
    setLoading(false);
  }

  const listCompaniesCallback = useCallback(getCompanies, [
    offset,
    developersPending,
    gamesPending,
  ]);

  async function searchCompanies(searchTerm: string) {
    setLoading(true);
    const searchResponse = await RBACClient.searchCompanies(
      new SearchCompaniesRequest({ query: searchTerm }),
    );
    setTotal(Math.ceil(searchResponse.Total / ORGANIZATION_LIMIT));
    setCompanies(searchResponse.companies);
    setLoading(false);
  }

  function searchCompaniesByName(term: string) {
    setSearch(term);
    if (term.length === 0) {
      listCompaniesCallback();
      return;
    }

    if (term.length < 4) {
      return;
    }

    debounceTimer.invoke(() => {
      searchCompanies(term);
    });
  }

  async function searchByGame(game?: Game) {
    if (!game) {
      return;
    }
    var result = await RBACClient.listCompanies(
      new ListCompaniesRequest({
        ownsGameId: parseInt(game.id, 10),
        limit: 1,
        offset: 0,
        hasDevelopersPending: developersPending,
        hasGamesPending: gamesPending,
      }),
    );

    setCompanies(result.companies);
    setTotal(Math.ceil(result.Total / ORGANIZATION_LIMIT));
    goToPage(1);
    setSearch('');
  }

  useEffect(() => {
    listCompaniesCallback();
  }, [listCompaniesCallback]);

  useEffect(() => {
    listCompaniesCallback();
  }, [gamesPending, page, developersPending, listCompaniesCallback]);

  return (
    <EntityList<Company>
      pagination={{
        onPreviousPage: previousPage,
        onNextPage: nextPage,
        onGoToPage: goToPage,
        currentPage: page,
        totalPages: total,
      }}
      loading={loading}
      items={companies}
      rowType={OrganizationListItem}
      rowProps={{}}
      columns={['Name', 'Organization ID', 'Created At', 'Type']}
      zeroStateMessage="No Companies found"
      header={{
        searchTerm: search,
        onSearchUpdate: searchCompaniesByName,
        searchPlaceholder: 'Search Organizations',
        additionalFields: [
          <GameAutocomplete onGameSelect={(game) => searchByGame(game)} />,
          <CheckBox
            label="Has Developers Pending?"
            checked={developersPending}
            onChange={(e: React.ChangeEvent<HTMLInputElement>) =>
              setDevelopersPending(e.target.checked)
            }
          />,
          <CheckBox
            label="Has Games Pending?"
            checked={gamesPending}
            onChange={(e: React.ChangeEvent<HTMLInputElement>) => setGamesPending(e.target.checked)}
          />,
        ],
      }}
    />
  );
};
