import React, { useState, useCallback, useEffect } from 'react';
import {
  CompanyApplication,
  SearchCompaniesRequest,
  Company,
} from '~core/clients/rbac/code.justin.tv.devrel';
import {
  Layout,
  Grid,
  StyledLayout,
  Background,
  Column,
  CoreText,
  TextType,
  Table,
  TableHeader,
  TableRow,
  TableHeading,
  TableBody,
  TableCell,
  Display,
  JustifyContent,
  SVGAsset,
  Color,
  Icon,
  AlignItems,
  BorderRadius,
  Button,
  ButtonType,
} from 'twitch-core-ui';
import { PageHeader } from '~features/page-header';
import { CompanyType } from '~features/organizations/models/organization-type';
import { OrganizationApplicationStatus } from '~features/organization-application/models/organization-application-status';
import './styles.scss';
import { RBACClient } from '~core/vienna';

export interface PublicProps {
  application: CompanyApplication;
}

type Props = PublicProps;

export const ApplicationProcessed: React.FC<Props> = (props) => {
  const [company, setCompany] = useState<Company>();
  async function getApprovedCompany() {
    try {
      const result = await RBACClient.searchCompanies(
        new SearchCompaniesRequest({
          query: props.application.companyName,
          limit: 1,
        }),
      );

      if (result.companies?.length) {
        setCompany(result.companies[0]);
      }
    } catch {
      // Nothing
    }
  }

  const getApprovedCompanyCallback = useCallback(getApprovedCompany, [
    props.application.companyName,
  ]);

  useEffect(() => {
    getApprovedCompanyCallback();
  }, [getApprovedCompanyCallback]);

  return (
    <Layout padding={2}>
      <Layout
        display={Display.Flex}
        justifyContent={JustifyContent.Between}
        alignItems={AlignItems.Start}
      >
        <PageHeader
          title={props.application.companyName}
          subtitle="Find all the details for this application below"
        />
        {props.application.status === OrganizationApplicationStatus.Rejected && (
          <StyledLayout
            color={Color.Overlay}
            className="application-status application-status--reject"
            display={Display.Flex}
            padding={2}
            borderRadius={BorderRadius.Large}
          >
            <StyledLayout color={Color.Overlay} margin={{ right: 1 }}>
              <Icon asset={SVGAsset.Halt} />
            </StyledLayout>
            <CoreText bold>Rejected</CoreText>
          </StyledLayout>
        )}
        {props.application.status === OrganizationApplicationStatus.Approved && (
          <StyledLayout
            color={Color.Base}
            className="application-status application-status--approved"
            display={Display.Flex}
            padding={2}
            borderRadius={BorderRadius.Large}
            alignItems={AlignItems.Center}
          >
            <StyledLayout color={Color.Base} margin={{ right: 1 }}>
              <Icon asset={SVGAsset.Check} />
            </StyledLayout>
            <Layout margin={{ right: company ? 2 : 0 }}>
              <CoreText bold>Approved</CoreText>
            </Layout>
            {company && (
              <Button type={ButtonType.Primary} overlay linkTo={`/organizations/${company.id}`}>
                Manage Organization
              </Button>
            )}
          </StyledLayout>
        )}
      </Layout>

      <StyledLayout background={Background.Base} padding={2} margin={{ bottom: 2 }}>
        <Grid>
          <Column cols={3}>
            <Layout margin={{ bottom: 1 }}>
              <CoreText bold>Organization Name</CoreText>
            </Layout>
            <CoreText>{props.application.companyName}</CoreText>
          </Column>
          <Column cols={3}>
            <Layout margin={{ bottom: 1 }}>
              <CoreText bold>URL</CoreText>
            </Layout>
            <CoreText>{props.application.companyWebsite}</CoreText>
          </Column>
          <Column cols={3}>
            <Layout margin={{ bottom: 1 }}>
              <CoreText bold>Type</CoreText>
            </Layout>
            <CoreText>{CompanyType[props.application.companyType]}</CoreText>
          </Column>
        </Grid>
      </StyledLayout>
      <Layout margin={{ bottom: 2 }}>
        <Layout margin={{ bottom: 2 }}>
          <CoreText type={TextType.H3}>Organization Owner</CoreText>
        </Layout>
        <Table>
          <TableHeader>
            <TableRow>
              <TableHeading label="First Name" />
              <TableHeading label="Last Name" />
              <TableHeading label="Title" />
              <TableHeading label="Email" />
              <TableHeading label="Email Verification Status" />
              <TableHeading label="Twitch ID" />
            </TableRow>
          </TableHeader>
          <TableBody>
            <TableRow>
              <TableCell>{props.application.contactFirstName}</TableCell>
              <TableCell>{props.application.contactLastName}</TableCell>
              <TableCell>{props.application.contactTitle}</TableCell>
              <TableCell>{props.application.contactEmail}</TableCell>
              <TableCell>{props.application.emailVerificationStatus}</TableCell>
              <TableCell>{props.application.twitchId}</TableCell>
            </TableRow>
          </TableBody>
        </Table>
      </Layout>
      {props.application.games && props.application.games.length > 0 && (
        <Layout>
          <Layout margin={{ bottom: 2 }}>
            <CoreText type={TextType.H3}>Game Applications</CoreText>
          </Layout>
          <Layout>
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHeading label="Game Name" />
                  <TableHeading label="Game Twitch ID" />
                </TableRow>
              </TableHeader>
              <TableBody>
                {props.application.games.map((game, index) => (
                  <TableRow key={index}>
                    <TableCell>{game.name}</TableCell>
                    <TableCell>{game.id}</TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </Layout>
        </Layout>
      )}
    </Layout>
  );
};
