import React, { useState, useEffect, useCallback } from 'react';
import { useParams } from 'react-router';
import { OrganizationUsersPresentation } from '~pages/organization-users/organization-users-presentation';
import {
  GetUsersByCompanyIdRequest,
  Membership,
  RemoveCompanyMembershipRequest,
  UpdateUserRequest,
} from '~core/clients/rbac/code.justin.tv.devrel';
import { RBACClient } from '~core/vienna';
import { toast } from 'react-toastify';
import { UserInformation } from '~features/organizations/models/user-information';
import { usePagination } from '~features/hooks/use-pagination';

export interface PublicProps {}

export interface RouteProps {
  organizationId: string;
}

const USERS_PER_PAGE = 25;

type Props = PublicProps;

export const OrganizationUsersPage: React.FC<Props> = () => {
  const [memberships, setMemberships] = useState<Membership[]>([]);
  const [total, setTotal] = useState<number>(0);

  const { page, previousPage, nextPage, goToPage } = usePagination();
  const { organizationId } = useParams<RouteProps>();

  async function removeUser(userId: string) {
    try {
      await RBACClient.removeCompanyMembership(
        new RemoveCompanyMembershipRequest({
          twitchId: userId,
          companyId: organizationId,
          requestingTwitchId: '',
        }),
      );

      toast('Successfully removed user', { type: 'success' });
      getCompanyMembers();
    } catch {
      toast('Error Removing User!', { type: 'warning' });
    }
  }

  async function getCompanyMembers() {
    try {
      const members = await RBACClient.getUsersByCompanyId(
        new GetUsersByCompanyIdRequest({
          id: organizationId,
          limit: USERS_PER_PAGE,
          offset: USERS_PER_PAGE * (page - 1),
        }),
      );
      setMemberships(members.memberships);
      setTotal(Math.ceil(members.total / USERS_PER_PAGE));
    } catch (err) {
      // Do nothing
    }
  }

  async function updateUser(user: UserInformation) {
    try {
      await RBACClient.updateUser(
        new UpdateUserRequest({
          twitchId: user.twitchUserId,
          companyId: organizationId,
          firstName: user.firstName,
          lastName: user.lastName,
          devEmail: user.email,
          devTitle: user.title,
        }),
      );
      getCompanyMembers();

      toast('Updated User!', { type: 'success' });
    } catch (e) {
      toast(`Error Updating User: ${e.message}`, { type: 'error' });
    }
  }

  const getCompanyMembersCallback = useCallback(getCompanyMembers, [organizationId, page]);

  useEffect(() => {
    getCompanyMembersCallback();
  }, [getCompanyMembersCallback]);

  return (
    <>
      <OrganizationUsersPresentation
        members={memberships}
        onRemoveUser={removeUser}
        onUpdateUser={updateUser}
        pagination={{
          currentPage: page,
          totalPages: total,
          onPreviousPage: previousPage,
          onNextPage: nextPage,
          onGoToPage: goToPage,
        }}
      />
    </>
  );
};
