const TsConfigPathsPlugin = require('tsconfig-paths-webpack-plugin');
const MiniCssExtractPlugin = require('mini-css-extract-plugin');
const HtmlWebpackPlugin = require('html-webpack-plugin');
const webpack = require('webpack');
const path = require('path');
const webpackbar = require('webpackbar');
const ForkTSCheckerPlugin = require('fork-ts-checker-webpack-plugin');
const { CleanWebpackPlugin } = require('clean-webpack-plugin');

const ROOT_PATH = path.normalize(__dirname);
const SRC_PATH = path.join(ROOT_PATH, 'src');
const PUBLIC_FOLDER = path.join(ROOT_PATH, 'public');
const INDEX_PATH = path.join(PUBLIC_FOLDER, 'index.html');

const filenamePrefix = 'assets/';
const filenamePostfix = '';

module.exports = (env, args) => {
  const mode = env.production ? 'production' : 'development';
  const production = mode === 'production';

  const devtool = production ? 'source-map' : 'inline-source-map';
  const resolve = {
    extensions: ['.ts', '.tsx', '.js', '.jsx'],
    plugins: [new TsConfigPathsPlugin({})],
  };

  const entry = {
    preboot: path.join(SRC_PATH, 'preboot.ts'),
    main: path.join(SRC_PATH, 'index.tsx'),
  };

  if (!production) {
    entry['dev'] = path.join(SRC_PATH, 'dev.setup.ts');
  }

  const output = {
    path: path.join(ROOT_PATH, 'dist'),
    filename: '[name].[hash].js',
    publicPath: '/',
  };

  const module = {
    rules: [
      {
        test: [/\.(ico|png|jpg|gif|svg|eot|ttf|woff|woff2|swf|webm|otf)$/],
        type: 'javascript/auto',
        loader: 'file-loader',
        options: {
          name: filenamePrefix + `[name]-[hash:hex:20]${filenamePostfix}.[ext]`,
        },
      },
      {
        test: /\.(graphql|gql)$/,
        exclude: /node_modules/,
        loader: 'graphql-tag/loader',
      },
      {
        test: /\.ts(x)?$/,
        use: [
          {
            loader: 'ts-loader',
            options: {
              transpileOnly: true,
            },
          },
        ],
      },
      {
        test: /\.s?[ac]ss$/,
        use: [
          {
            loader: mode === 'development' ? 'style-loader' : MiniCssExtractPlugin.loader,
          },
          {
            loader: 'css-loader',
            options: {
              sourceMap: true,
            },
          },
          {
            loader: 'postcss-loader',
            options: {
              sourceMap: true,
            },
          },
          {
            loader: 'sass-loader',
            options: {
              sourceMap: true,
              sassOptions: {
                includePaths: ['node_modules/twitch-core-ui'],
              },
            },
          },
        ],
      },
    ],
  };

  const devServer = {
    port: 3000,
    contentBase: PUBLIC_FOLDER,
    hot: true,
    historyApiFallback: true,
  };

  const plugins = [
    new HtmlWebpackPlugin({
      template: INDEX_PATH,
      favicon: PUBLIC_FOLDER + '/favicon.ico',
    }),
    new webpack.HotModuleReplacementPlugin(),
    new webpackbar({
      color: 'green',
    }),
    new ForkTSCheckerPlugin({
      eslint: true,
      checkSyntacticErrors: true,
    }),
    new webpack.DefinePlugin({
      // Production mode is for running on staging or production environment.
      PRODUCTION: JSON.stringify(mode !== 'development'),

      // Vienna edge can be local, staging or production when running on a local environment.
      // By default is empty (''), targeting the same origin when served from the Go edge app as a static asset.
      EDGE_HOST: JSON.stringify(env.EDGE_HOST || ''),
    }),
  ];

  if (production) {
    plugins.push(
      new MiniCssExtractPlugin({
        filename: '[name].[hash].css',
      }),
    );
    plugins.push(new CleanWebpackPlugin());
  }

  const performance = {
    hints: false,
  };

  return {
    mode,
    entry,
    output,
    devtool,
    resolve,
    plugins,
    module,
    performance: production ? performance : undefined,
    devServer: mode === 'development' ? devServer : undefined,
    stats: production ? 'normal' : 'minimal',
  };
};
