package actionhistories

import (
	"context"
	"database/sql"
	"fmt"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/dbx"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

const (
	// RBAC specific developer guid that represents no user
	AnonymousUserUUID     = "00000000-0000-0000-0000-000000000000"
	AnonymousUserTwitchID = -1

	Table = "action_histories"
)

// ActionHistories manages the action_histories DB table.
//go:generate errxer --timings ActionHistories
//go:generate counterfeiter . ActionHistories
type ActionHistories interface {
	ListActionHistories(ctx context.Context, params ListParams) ([]*rbacrpc.ActionHistory, uint64, error)
	InsertActionHistory(ctx context.Context, a *ActionHistory)
	StripUserActionHistory(ctx context.Context, userID string) error
}

// ActionHistory represents to one row in the action_histories table
type ActionHistory struct {
	ID           string         `db:"id"`
	UserTwitchID string         `db:"user_twitch_id"`
	Action       string         `db:"action"`      // english text with a description of what happened
	EntityType   string         `db:"entity_type"` // e.g. "Game"
	EntityID     string         `db:"entity_id"`   // sometimes the id, sometimes the identifier ...
	CreatedAt    string         `db:"created_at"`
	CompanyID    sql.NullString `db:"company_id"`

	XXX_Total uint64 `db:"total"`
}

var Columns = dbx.FieldsFrom(ActionHistory{}).Exclude("total")

func New(dbxDB common.DBXer, stats statsd.Statter) ActionHistories {
	impl := &DBXActionHistories{DBX: dbxDB}
	wrapper := &ActionHistoriesErrx{
		ActionHistories: impl,
		TimingFunc:      common.TimingStats(stats),
	}
	return wrapper
}

type DBXActionHistories struct {
	DBX common.DBXer
}

type ListParams struct {
	CompanyID  string
	EntityID   string
	EntityType string
	Limit      uint64
	Offset     uint64
}

func (d *DBXActionHistories) ListActionHistories(ctx context.Context, params ListParams) ([]*rbacrpc.ActionHistory, uint64, error) {
	var list []*ActionHistory

	q := common.PSQL.Select(Columns.Add(common.CountOverAs("total"))...).From(Table)
	if params.CompanyID != "" {
		q = q.Where("company_id = ?", params.CompanyID)
	}

	if params.EntityID != "" {
		q = q.Where("entity_id = ?", params.EntityID)
	}

	if params.EntityType != "" {
		q = q.Where("entity_type = ?", params.EntityType)
	}

	q = common.Paginate(q, params.Limit, params.Offset).OrderBy("created_at DESC")

	err := d.DBX.LoadAll(ctx, &list, q)
	return ConvertActionHistoriesListToRPC(list), common.FirstRowUInt64DBField(list, "total"), err
}

func (d *DBXActionHistories) InsertActionHistory(ctx context.Context, a *ActionHistory) {
	a.ID = common.NewUUID()
	a.CreatedAt = common.TimeNowStr()

	err := d.DBX.InsertOne(ctx, Table, a, dbx.Exclude("total"))

	// In case of error, we report this to ourselves and log all the fields so we can
	// recover the lost action in DB, but we don't stop the action from happening.
	if err != nil {
		logx.Error(ctx, err, a.ToLogxFields())
	}
}

func (d *DBXActionHistories) StripUserActionHistory(ctx context.Context, userID string) error {
	_, err := d.DBX.NamedExec(ctx, fmt.Sprintf("UPDATE %s SET user_twitch_id = :newUserID WHERE user_twitch_id = :oldUserID", Table),
		map[string]interface{}{
			"oldUserID": userID,
			"newUserID": AnonymousUserTwitchID,
		})

	return err
}

//
// Converters
//

func (a *ActionHistory) ToRPC() *rbacrpc.ActionHistory {
	companyID := ""
	if a.CompanyID.Valid {
		companyID = a.CompanyID.String
	}

	return &rbacrpc.ActionHistory{
		Id:           a.ID,
		UserTwitchId: a.UserTwitchID,
		Action:       a.Action,
		EntityType:   a.EntityType,
		EntityId:     a.EntityID,
		CreatedAt:    a.CreatedAt,
		CompanyId:    companyID,
	}
}

func ConvertActionHistoriesListToRPC(list []*ActionHistory) []*rbacrpc.ActionHistory {
	ret := make([]*rbacrpc.ActionHistory, len(list))
	for i, a := range list {
		ret[i] = a.ToRPC()
	}
	return ret
}

func (a *ActionHistory) ToLogxFields() logx.Fields {
	return logx.Fields{
		"id":             a.ID,
		"user_twitch_id": a.UserTwitchID,
		"action":         a.Action,
		"entity_type":    a.EntityType,
		"entity_id":      a.EntityID,
		"created_at":     a.CreatedAt,
		"company_id":     a.CompanyID,
	}
}
