package backend

import (
	"context"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/dbx"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/backend/companies"
	"code.justin.tv/devrel/devsite-rbac/backend/companyapplications"
	"code.justin.tv/devrel/devsite-rbac/backend/companyinvites"
	"code.justin.tv/devrel/devsite-rbac/backend/companyresources"
	"code.justin.tv/devrel/devsite-rbac/backend/extensionbillingmanagers"
	"code.justin.tv/devrel/devsite-rbac/backend/gameapplications"

	"github.com/cactus/go-statsd-client/statsd"
)

// Config sets options for a backend handler
type Config struct {
	DB    common.DBXer
	Stats statsd.Statter
}

type backend struct {
	db common.DBXer

	companyresources.CRBackender
	companies.CBackender
	companyinvites.CompanyInvites
	extensionbillingmanagers.ExtensionBillingManagers
	gameapplications.GABackender
	companyapplications.CompanyApplications
}

// Backender implements the interface for reading/writing models from/to disk.
//go:generate counterfeiter . Backender
type Backender interface {
	// Begin returns a Backender that uses a transaction
	Begin(ctx context.Context) context.Context
	// Commit commits the transaction in use, if available
	Commit(context.Context) error
	// Rollback rolls back the transaction in use, if available. The provided error describes why rollback is occurring.
	Rollback(context.Context)

	companyresources.CRBackender
	companies.CBackender
	companyinvites.CompanyInvites
	extensionbillingmanagers.ExtensionBillingManagers
	gameapplications.GABackender
	companyapplications.CompanyApplications
}

// NewBackend instantiates a new backend handler
func NewBackend(c *Config) (Backender, error) {
	db := c.DB
	return &backend{
		db:                       db,
		CRBackender:              companyresources.New(db, c.Stats),
		CBackender:               companies.New(db, c.Stats),
		CompanyInvites:           companyinvites.New(db, c.Stats),
		GABackender:              gameapplications.New(db, c.Stats),
		CompanyApplications:      companyapplications.New(db, c.Stats),
		ExtensionBillingManagers: extensionbillingmanagers.New(db, c.Stats),
	}, nil
}

func (b *backend) Begin(ctx context.Context) context.Context {
	var impl common.DBXer = b.db
	errxer, ok := b.db.(*common.DBXerErrx)
	if ok {
		impl = errxer.DBXer
	}

	// panic if this fails
	dbxer := impl.(*dbx.DBX)

	return dbx.MustBegin(ctx, dbxer.DB)
}

func (b *backend) Commit(ctx context.Context) error {
	return dbx.Commit(ctx)
}

func (b *backend) Rollback(ctx context.Context) {
	dbx.RollbackUnlessComitted(ctx, func(err error) {
		logx.Error(ctx, err)
	})
}
