package companyapplications_test

import (
	"context"
	"database/sql"
	"testing"

	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/stretchr/testify/require"

	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/backend/companyapplications"
	"code.justin.tv/devrel/devsite-rbac/backend/companyapplications/companyapplicationsfakes"
	"code.justin.tv/devrel/devsite-rbac/backend/testutils"
)

func TestGetCompanyApplication(t *testing.T) {
	fakeBackend := &companyapplicationsfakes.FakeCompanyApplications{}
	fakeSecureBackend := &companyapplications.SecureBackend{
		CompanyApplications: fakeBackend,
	}
	testCompanyId := common.NewUUID()
	ctx := testutils.MockContextWithOauthToken(context.Background())

	t.Run("VIEWER should not see secure fields", func(t *testing.T) {
		ctx = testutils.MockContextWithViennaRole(ctx, rbacrpc.WhitelistUserRole_VIEWER)
		capp := newCompanyApplication(testCompanyId)
		fakeBackend.GetCompanyApplicationReturns(capp, nil)

		result, err := fakeSecureBackend.GetCompanyApplication(ctx, testCompanyId)

		require.NoError(t, err)
		require.NotNil(t, result)
		require.NotEmpty(t, result.Id)
		require.Empty(t, result.ContactFirstName)
		require.Empty(t, result.ContactLastName)
		require.Empty(t, result.ContactEmail)
	})

	t.Run("SECURE_VIEWER should see secure fields", func(t *testing.T) {
		ctx = testutils.MockContextWithViennaRole(ctx, rbacrpc.WhitelistUserRole_SECURE_VIEWER)
		capp := newCompanyApplication(testCompanyId)
		fakeBackend.GetCompanyApplicationReturns(capp, nil)

		result, err := fakeSecureBackend.GetCompanyApplication(ctx, testCompanyId)

		require.NoError(t, err)
		require.NotNil(t, result)
		require.NotEmpty(t, result.Id)
		require.NotEmpty(t, result.ContactFirstName)
		require.NotEmpty(t, result.ContactLastName)
		require.NotEmpty(t, result.ContactEmail)
	})

	t.Run("ADMIN should see secure fields", func(t *testing.T) {
		ctx = testutils.MockContextWithViennaRole(ctx, rbacrpc.WhitelistUserRole_ADMIN)
		capp := newCompanyApplication(testCompanyId)
		fakeBackend.GetCompanyApplicationReturns(capp, nil)

		result, err := fakeSecureBackend.GetCompanyApplication(ctx, testCompanyId)

		require.NoError(t, err)
		require.NotNil(t, result)
		require.NotEmpty(t, result.Id)
		require.NotEmpty(t, result.ContactFirstName)
		require.NotEmpty(t, result.ContactLastName)
		require.NotEmpty(t, result.ContactEmail)
	})
}

func TestListCompanyApplications(t *testing.T) {
	fakeBackend := &companyapplicationsfakes.FakeCompanyApplications{}
	fakeSecureBackend := &companyapplications.SecureBackend{
		CompanyApplications: fakeBackend,
	}
	testCompanyId := common.NewUUID()
	ctx := testutils.MockContextWithOauthToken(context.Background())

	t.Run("VIEWER should not see secure fields", func(t *testing.T) {
		ctx = testutils.MockContextWithViennaRole(ctx, rbacrpc.WhitelistUserRole_VIEWER)
		capp := newCompanyApplication(testCompanyId)
		fakeBackend.ListCompanyApplicationsReturns([]companyapplications.CompanyApplication{capp}, 1, nil)

		list, _, err := fakeSecureBackend.ListCompanyApplications(ctx, &rbacrpc.ListCompanyApplicationsRequest{
			TwitchId: "test_twitch_id",
			Limit:    15,
			Offset:   0,
		})

		require.NoError(t, err)
		require.NotEmpty(t, list)
		require.Empty(t, list[0].ContactFirstName)
		require.Empty(t, list[0].ContactLastName)
		require.Empty(t, list[0].ContactEmail)
	})

	t.Run("SECURE_VIEWER should see secure fields", func(t *testing.T) {
		ctx = testutils.MockContextWithViennaRole(ctx, rbacrpc.WhitelistUserRole_SECURE_VIEWER)
		capp := newCompanyApplication(testCompanyId)
		fakeBackend.ListCompanyApplicationsReturns([]companyapplications.CompanyApplication{capp}, 1, nil)

		list, _, err := fakeSecureBackend.ListCompanyApplications(ctx, &rbacrpc.ListCompanyApplicationsRequest{
			TwitchId: "test_twitch_id",
			Limit:    15,
			Offset:   0,
		})

		require.NoError(t, err)
		require.NotEmpty(t, list)
		require.NotEmpty(t, list[0].ContactFirstName)
		require.NotEmpty(t, list[0].ContactLastName)
		require.NotEmpty(t, list[0].ContactEmail)
	})

	t.Run("ADMIN should see secure fields", func(t *testing.T) {
		ctx = testutils.MockContextWithViennaRole(ctx, rbacrpc.WhitelistUserRole_ADMIN)
		capp := newCompanyApplication(testCompanyId)
		fakeBackend.ListCompanyApplicationsReturns([]companyapplications.CompanyApplication{capp}, 1, nil)

		list, _, err := fakeSecureBackend.ListCompanyApplications(ctx, &rbacrpc.ListCompanyApplicationsRequest{
			TwitchId: "test_twitch_id",
			Limit:    15,
			Offset:   0,
		})

		require.NoError(t, err)
		require.NotEmpty(t, list)
		require.NotEmpty(t, list[0].ContactFirstName)
		require.NotEmpty(t, list[0].ContactLastName)
		require.NotEmpty(t, list[0].ContactEmail)
	})
}

func TestSearchCompanyApplications(t *testing.T) {
	fakeBackend := &companyapplicationsfakes.FakeCompanyApplications{}
	fakeSecureBackend := &companyapplications.SecureBackend{
		CompanyApplications: fakeBackend,
	}
	testCompanyId := common.NewUUID()
	ctx := testutils.MockContextWithOauthToken(context.Background())

	t.Run("VIEWER should not see secure fields", func(t *testing.T) {
		ctx = testutils.MockContextWithViennaRole(ctx, rbacrpc.WhitelistUserRole_VIEWER)
		capp := newCompanyApplication(testCompanyId)
		fakeBackend.SearchCompanyApplicationsReturns([]companyapplications.CompanyApplication{capp}, 1, nil)

		list, _, err := fakeSecureBackend.SearchCompanyApplications(ctx, "ss", 15)
		require.NoError(t, err)
		require.NotEmpty(t, list)
		require.Empty(t, list[0].ContactFirstName)
		require.Empty(t, list[0].ContactLastName)
		require.Empty(t, list[0].ContactEmail)
	})

	t.Run("SECURE_VIEWER should see secure fields", func(t *testing.T) {
		ctx = testutils.MockContextWithViennaRole(ctx, rbacrpc.WhitelistUserRole_SECURE_VIEWER)
		capp := newCompanyApplication(testCompanyId)
		fakeBackend.SearchCompanyApplicationsReturns([]companyapplications.CompanyApplication{capp}, 1, nil)

		list, _, err := fakeSecureBackend.SearchCompanyApplications(ctx, "ss", 15)
		require.NoError(t, err)
		require.NotEmpty(t, list)
		require.NotEmpty(t, list[0].ContactFirstName)
		require.NotEmpty(t, list[0].ContactLastName)
		require.NotEmpty(t, list[0].ContactEmail)
	})

	t.Run("ADMIN should see secure fields", func(t *testing.T) {
		ctx = testutils.MockContextWithViennaRole(ctx, rbacrpc.WhitelistUserRole_ADMIN)
		capp := newCompanyApplication(testCompanyId)
		fakeBackend.SearchCompanyApplicationsReturns([]companyapplications.CompanyApplication{capp}, 1, nil)

		list, _, err := fakeSecureBackend.SearchCompanyApplications(ctx, "ss", 15)
		require.NoError(t, err)
		require.NotEmpty(t, list)
		require.NotEmpty(t, list[0].ContactFirstName)
		require.NotEmpty(t, list[0].ContactLastName)
		require.NotEmpty(t, list[0].ContactEmail)
	})
}

//
// Test Helpers
//

func newCompanyApplication(companyId string) companyapplications.CompanyApplication {
	return companyapplications.CompanyApplication{
		Id:               companyId,
		TwitchId:         "1122334455",
		Status:           1, // pending
		CompanyName:      "test_company_name",
		CompanyWebsite:   "https://testcompany.com",
		CompanyType:      1,
		CompanySize:      "50",
		City:             "Los Angeles",
		State:            "CA",
		Country:          "US",
		ContactTitle:     "developer",
		Industry:         "Games",
		ProductInterest:  sql.NullString{Valid: true, String: "Insights"},
		JoinReason:       sql.NullString{Valid: true, String: "Insights"},
		ContactFirstName: "test_contact_firstname",
		ContactLastName:  "test_contact_lastname",
		ContactEmail:     "test@test.com",
	}
}
