package companyinvites

import (
	"context"
	"fmt"
	"time"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/devrel/dbx"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

const (
	Table             = "company_invites"
	ExpirationInHours = 72 // If set to 0 the check will not be performed
)

//go:generate counterfeiter . CompanyInvites
//go:generate errxer --timings CompanyInvites
type CompanyInvites interface {
	SelectCompanyInvite(ctx context.Context, companyInviteId string) (*rbacrpc.CompanyInvite, error)
	InsertCompanyInvite(ctx context.Context, i *CompanyInvite) (*rbacrpc.CompanyInvite, error)
	DeleteCompanyInvite(ctx context.Context, companyInviteId string) error
	DeleteAllCompanyInvitesByTwitchId(ctx context.Context, twitchId string) error
	SelectComanyInvitesByInviteeTwitchId(ctx context.Context, twitchId string, limit uint64, offset uint64) ([]*rbacrpc.CompanyInvite, int32, error)
	SelectComanyInvitesByCompanyId(ctx context.Context, companyId string, limit uint64, offset uint64) ([]*rbacrpc.CompanyInvite, int32, error)
	AnyCompanyInvitesForUserOnOrganization(ctx context.Context, companyId string, inviteeTwitchId string) (bool, error)
	ExpiredCompanyInviteForUserOnOrganization(ctx context.Context, companyId string, inviteeTwitchId string) (string, error)
}

type CompanyInvite struct {
	ID              string `db:"id"`
	CompanyID       string `db:"company_id"`
	InviteeTwitchID string `db:"invitee_twitch_id"`
	InviterTwitchID string `db:"inviter_twitch_id"`
	Role            string `db:"role"`

	CreatedAt string `db:"created_at"`
	XXX_Total int32  `db:"_total"`
}

var Columns = dbx.FieldsFrom(&CompanyInvite{}).Exclude("_total")

type DBXCompanyInvites struct {
	db common.DBXer
}

func New(db common.DBXer, stats statsd.Statter) CompanyInvites {
	return &CompanyInvitesErrx{
		CompanyInvites: &DBXCompanyInvites{db: db},
		TimingFunc:     common.TimingStats(stats),
	}
}

func (d *DBXCompanyInvites) SelectCompanyInvite(ctx context.Context, companyInviteId string) (*rbacrpc.CompanyInvite, error) {
	q := common.PSQL.Select(Columns...).From(Table).Where("id = ?", companyInviteId)
	var c CompanyInvite
	err := d.db.LoadOne(ctx, &c, q)
	return c.ToRPC(), err
}

func (d *DBXCompanyInvites) SelectComanyInvitesByInviteeTwitchId(ctx context.Context, twitchId string, limit uint64, offset uint64) ([]*rbacrpc.CompanyInvite, int32, error) {
	cols := append(Columns, "count(*) OVER() as _total")
	q := common.PSQL.Select(cols...).
		From(Table).Where("invitee_twitch_id = ?", twitchId).OrderBy("created_at").Limit(limit).Offset(offset)

	var invites []CompanyInvite
	err := d.db.LoadAll(ctx, &invites, q)
	if err != nil {
		return nil, 0, err
	}

	return ListToRPC(invites), common.FirstRowInt32DBField(invites, "_total"), err
}

func (d *DBXCompanyInvites) DeleteAllCompanyInvitesByTwitchId(ctx context.Context, twitchId string) error {
	ci := CompanyInvite{
		InviteeTwitchID: twitchId,
	}

	_, err := d.db.NamedExec(ctx, fmt.Sprintf("DELETE FROM %s WHERE invitee_twitch_id = :invitee_twitch_id", Table), ci)

	return err
}

func (d *DBXCompanyInvites) SelectComanyInvitesByCompanyId(ctx context.Context, companyId string, limit uint64, offset uint64) ([]*rbacrpc.CompanyInvite, int32, error) {
	cols := append(Columns, "count(*) OVER() as _total")
	q := common.PSQL.Select(cols...).
		From(Table).Where("company_id = ?", companyId).OrderBy("created_at").Limit(limit).Offset(offset)

	var invites []CompanyInvite
	err := d.db.LoadAll(ctx, &invites, q)
	if err != nil {
		return nil, 0, err
	}

	return ListToRPC(invites), common.FirstRowInt32DBField(invites, "_total"), err
}

func (d *DBXCompanyInvites) DeleteCompanyInvite(ctx context.Context, companyInviteId string) error {
	return d.db.DeleteOne(ctx, Table, dbx.Values{"id": companyInviteId})
}

func (d *DBXCompanyInvites) InsertCompanyInvite(ctx context.Context, i *CompanyInvite) (*rbacrpc.CompanyInvite, error) {
	id := common.NewUUID()

	companyInvite := &CompanyInvite{
		ID:              id,
		CompanyID:       i.CompanyID,
		InviteeTwitchID: i.InviteeTwitchID,
		InviterTwitchID: i.InviterTwitchID,
		Role:            i.Role,
		CreatedAt:       common.TimeNowStr(),
	}
	err := d.db.InsertOne(ctx, Table, companyInvite, dbx.Exclude("_total"))
	return companyInvite.ToRPC(), err
}

func (d *DBXCompanyInvites) AnyCompanyInvitesForUserOnOrganization(ctx context.Context, companyId string, inviteeTwitchId string) (bool, error) {
	q := common.PSQL.Select(Columns...).From(Table).Where("company_id = ?", companyId).Where("invitee_twitch_id = ?", inviteeTwitchId)
	var c CompanyInvite
	err := d.db.LoadOne(ctx, &c, q)

	if errorutil.IsErrNoRows(err) {
		return false, nil
	}

	if err != nil {
		return false, err
	}

	return true, nil

}

func (d *DBXCompanyInvites) ExpiredCompanyInviteForUserOnOrganization(ctx context.Context, companyId string, inviteeTwitchId string) (string, error) {

	q := common.PSQL.Select(Columns...).From(Table).Where("company_id = ?", companyId).Where("invitee_twitch_id = ?", inviteeTwitchId).Where("created_at+interval '72 hours' < NOW()")
	var c CompanyInvite
	err := d.db.LoadOne(ctx, &c, q)

	if errorutil.IsErrNoRows(err) {
		return "", nil
	}

	if err != nil {
		return "", err
	}
	return c.ID, nil
}

//
// Converters
//

func (i CompanyInvite) ToRPC() *rbacrpc.CompanyInvite {
	return &rbacrpc.CompanyInvite{
		Id:              i.ID,
		CompanyId:       i.CompanyID,
		InviteeTwitchId: i.InviteeTwitchID,
		InviterTwitchId: i.InviterTwitchID,
		Role:            i.Role,
		CreatedAt:       i.CreatedAt,
		ExpiresAt:       i.ExpiresAt(),
	}
}

func ListToRPC(list []CompanyInvite) []*rbacrpc.CompanyInvite {
	invites := make([]*rbacrpc.CompanyInvite, len(list))
	for i, invite := range list {
		invites[i] = invite.ToRPC()
	}
	return invites
}

func (i CompanyInvite) ExpiresAt() string {
	dateCreated, err := time.Parse("2006-01-02T15:04:05Z07:00", i.CreatedAt)
	if err != nil {
		return ""
	}
	return dateCreated.Add(ExpirationInHours * time.Hour).Format(time.RFC3339)
}
