package cartman

import (
	"context"
	"errors"
	"io/ioutil"
	"net/http"
	"strings"
	"testing"

	"code.justin.tv/foundation/twitchclient"

	"github.com/stretchr/testify/require"
)

func TestGetToken_Success(t *testing.T) {
	httpCli := fakeHTTPClient{respBody: `{
		"token": "returned-cartman-token",
		"user_id": "123"
	}`}
	c := &client{JSONClient: twitchclient.NewJSONClient("http://cartman.tv", &httpCli)}
	authHeader := "OAuth fkkfkkfkkfkkOAuthtoken"
	capabilities := "extensions::view_all_extensions,extensions::edit_version"
	authFnParams := map[string]string{"extension_id": "extid1234"}

	cartmanToken, err := c.GetToken(ctx, authHeader, capabilities, authFnParams)
	require.NoError(t, err)

	url := httpCli.sentReq.URL
	require.Equal(t, "cartman.tv", url.Host)
	require.Equal(t, "/authorization_token", url.Path)
	require.Equal(t, "returned-cartman-token", cartmanToken)
	require.Equal(t, "extid1234", url.Query().Get("extension_id"))
	require.Equal(t, capabilities, url.Query().Get("capabilities"))
	require.Equal(t, authHeader, httpCli.sentReq.Header.Get("Authorization"))
}

func TestGetToken_NilAuthFnParams(t *testing.T) {
	httpCli := fakeHTTPClient{respBody: `{
		"token": "returned-cartman-token",
		"user_id": "123"
	}`}
	c := &client{JSONClient: twitchclient.NewJSONClient("http://cartman.tv", &httpCli)}

	// Using authFnParams = nil works like an empty map
	cartmanToken, err := c.GetToken(ctx, "oauth", "cartman::authenticate_first_party", nil)
	require.NoError(t, err)
	require.Equal(t, "returned-cartman-token", cartmanToken)
	url := httpCli.sentReq.URL
	require.Equal(t, "", url.Query().Get("non-existing-parameter"))
	require.Equal(t, "cartman::authenticate_first_party", url.Query().Get("capabilities"))
}

func TestGetToken_InternalError(t *testing.T) {
	httpCli := fakeHTTPClient{status: 500, respBody: `{"message": "Internal Error"}`}
	c := &client{JSONClient: twitchclient.NewJSONClient("http://cartman.tv", &httpCli)}

	cartmanToken, err := c.GetToken(ctx, "oauth", "extensions::view_all_extensions", map[string]string{"foo": "bar"})
	require.Equal(t, 500, twitchclient.StatusCode(err))
	require.Equal(t, "", cartmanToken)
}

func TestInitialize_NetworkError(t *testing.T) {
	httpCli := fakeHTTPClient{err: errors.New("Fake network error")}
	c := &client{JSONClient: twitchclient.NewJSONClient("http://cartman.tv", &httpCli)}

	cartmanToken, err := c.GetToken(ctx, "oauth", "cartman::authenticate_first_party", nil)
	require.EqualError(t, err, "Fake network error")
	require.Equal(t, "", cartmanToken)
}

// ----------
// Test Utils
// ----------

var ctx = context.Background()

// fakeHTTPClient can be configured to fake different status and errors
type fakeHTTPClient struct {
	status   int    // response status, default 200.
	respBody string // response body as string.
	err      error  // response HTTP error, default nil.

	sentReq *http.Request // can be used after calling .Do for inspection.
}

// Do makes a fake request that depends on the configured fake attrs.
func (c *fakeHTTPClient) Do(req *http.Request) (*http.Response, error) {
	c.sentReq = req

	if c.err != nil {
		return nil, c.err
	}

	if c.status == 0 {
		c.status = 200
	}

	body := ioutil.NopCloser(strings.NewReader(c.respBody))
	return &http.Response{StatusCode: c.status, Body: body}, nil
}
