package channels_client

import (
	"context"
	"strconv"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/web/users-service/client/channels"
	"code.justin.tv/web/users-service/models"
)

// Channels is a simplied interface for the User Service Channels(https://git-aws.internal.justin.tv/web/users-service/blob/master/client/channels/client.go)
//go:generate errxer Channels
//go:generate counterfeiter . Channels
type Channels interface {
	HideChannel(ctx context.Context, twitchID string) error
}

type channelsImpl struct {
	client channels.Client
}

func NewChannels(clientConf twitchclient.ClientConf) (Channels, error) {
	client, err := channels.NewClient(clientConf)
	if err != nil {
		return nil, err
	}
	return &ChannelsErrx{Channels: &channelsImpl{client: client}}, nil
}

// HideChannel hides the channel from Search and Discovery by flipping "DirectoryHidden" and "PrivacyOptionsEnabled" to true
func (c *channelsImpl) HideChannel(ctx context.Context, twitchID string) error {
	// set the following 2 fields to true so that the channel can be hidden from Search and Discovery
	twitchIDUint, err := strconv.ParseUint(twitchID, 10, 64)
	if err != nil {
		return err
	}
	directoryHidden := true
	privacyOptionsEnabled := true

	up := models.UpdateChannelProperties{
		ID:                    twitchIDUint,
		DirectoryHidden:       &directoryHidden,
		PrivacyOptionsEnabled: &privacyOptionsEnabled,
	}
	return c.client.Set(ctx, up, defaultReqOpts())
}

func defaultReqOpts() *twitchclient.ReqOpts {
	return &twitchclient.ReqOpts{
		StatSampleRate: 1,
	}
}
