package cryptionclient

import (
	"crypto/aes"
	"crypto/cipher"
	"crypto/rand"

	"code.justin.tv/devrel/devsite-rbac/clients/kmsclient"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/kms"
)

type CryptionClient interface {
	EncryptPlainText(plaintext []byte) (encryptedDataKey *kmsclient.EncryptedDataKey, ciphertext []byte, err error)
	DecryptPlainText(dataKey *kmsclient.EncryptedDataKey, ciphertext []byte) (string, error)
}

type Client struct {
	KmsClient kmsclient.KmsClient
}

func (c *Client) EncryptPlainText(plaintext []byte) (encryptedDataKey *kmsclient.EncryptedDataKey, ciphertext []byte, err error) {
	dataKeyOutput, err := c.KmsClient.GenerateDataKey()
	if err != nil {
		return nil, nil, err
	}

	// override field GenerateDataKey.Plaintext with encrypted text, recommended by: https://docs.aws.amazon.com/sdk-for-go/api/service/kms/#GenerateDataKeyOutput
	dataKeyOutput.Plaintext, err = encrypt(dataKeyOutput.Plaintext, plaintext, []byte(aws.StringValue(dataKeyOutput.KeyId)))
	if err != nil {
		return nil, nil, err
	}

	return &kmsclient.EncryptedDataKey{
		Value:  dataKeyOutput.CiphertextBlob,
		Region: c.KmsClient.GetRegion(),
		KeyARN: aws.StringValue(dataKeyOutput.KeyId),
	}, dataKeyOutput.Plaintext, nil
}

func (c *Client) DecryptPlainText(dataKey *kmsclient.EncryptedDataKey, ciphertext []byte) (string, error) {
	decryptOutput, err := c.KmsClient.Decrypt(&kms.DecryptInput{
		CiphertextBlob: dataKey.Value,
	})
	if err != nil {
		return "", err
	}

	plaintextInByteArray, err := decrypt(decryptOutput.Plaintext, ciphertext, []byte(dataKey.KeyARN))
	if err != nil {
		return "", err
	}

	return string(plaintextInByteArray), nil
}

func encrypt(key, plaintext, data []byte) ([]byte, error) {
	block, err := aes.NewCipher(key)
	if err != nil {
		return nil, err
	}

	gcm, err := cipher.NewGCM(block)
	if err != nil {
		return nil, err
	}

	nonce := make([]byte, gcm.NonceSize())
	if _, err := rand.Read(nonce); err != nil {
		return nil, err
	}

	return gcm.Seal(nonce, nonce, plaintext, data), nil
}

func decrypt(key, ciphertext, data []byte) ([]byte, error) {
	block, err := aes.NewCipher(key)
	if err != nil {
		return nil, err
	}
	gcm, err := cipher.NewGCM(block)
	if err != nil {
		return nil, err
	}
	nonce, ciphertext := ciphertext[:gcm.NonceSize()], ciphertext[gcm.NonceSize():]

	return gcm.Open(nil, nonce, ciphertext, data)
}

func NewClient(keyARN string, region string) (CryptionClient, error) {
	kms, err := kmsclient.NewClient(keyARN, region)
	if err != nil {
		return new(Client), err
	}
	return &Client{
		KmsClient: kms,
	}, nil
}
