package dart

import (
	"context"
	"fmt"

	"code.justin.tv/foundation/twitchclient"
	receiver "code.justin.tv/amzn/TwitchDartReceiverTwirp"
	"code.justin.tv/identity/passport/logger"
	"github.com/pkg/errors"
)

// Dart is new notifications platform we will be using to send emails
//go:generate errxer Dart
//go:generate counterfeiter . Dart
type Dart interface {
	SendCompanyInviteToUser(ctx context.Context, userId string, organizationName string) (string, error)
	SendDeveloperCompanyActionApproved(ctx context.Context, userId string, organizationName string) (string, error)
	SendDeveloperCompanyActionDenied(ctx context.Context, userId string, organizationName string) (string, error)
	SendDeveloperCompanyUserActionAdded(ctx context.Context, userId string, organizationName string) (string, error)
	SendDeveloperCompanyUserActionRemoved(ctx context.Context, userId string, organizationName string) (string, error)
	SendDeveloperCompanyGameActionApproved(ctx context.Context, userId string, gameName string) (string, error)
	SendDeveloperCompanyGameActionDenied(ctx context.Context, userId string, gameName string) (string, error)
	PublishNotification(ctx context.Context, request *receiver.PublishNotificationRequest) (string, error)
}

type dartImpl struct {
	client receiver.Receiver
}

const (
	DevsiteCompanyInvite               = "devsite_company_invite"
	DeveloperCompanyActionApproved     = "developer_company_action_approved"
	DeveloperCompanyActionDenied       = "developer_company_action_denied"
	DeveloperCompanyUserActionAdded    = "developer_company_user_action_added"
	DeveloperCompanyUserActionRemoved  = "developer_company_user_action_removed"
	DeveloperCompanyGameActionApproved = "developer_company_game_action_approved"
	DeveloperCompanyGameActionDenied   = "developer_company_game_action_denied"
)

func New(host string) (Dart, error) {
	receiver := receiver.NewReceiverProtobufClient(host,
		twitchclient.NewHTTPClient(twitchclient.ClientConf{
			Host: host,
		}))
	return &DartErrx{Dart: &dartImpl{client: receiver}}, nil

}

func (r *dartImpl) PublishNotification(ctx context.Context, request *receiver.PublishNotificationRequest) (string, error) {
	resp, publishErr := r.client.PublishNotification(ctx, request)
	if publishErr != nil {
		errorMessage := fmt.Sprintf("failed to publish %s notification to dart: %v", request.NotificationType, publishErr)
		logger.Log(ctx, logger.Err, errorMessage)
		return "", errors.Wrap(publishErr, errorMessage)
	}
	return resp.NotificationTraceId, nil
}

func (r dartImpl) SendCompanyInviteToUser(ctx context.Context, userId string, organizationName string) (string, error) {

	dartRequest := &receiver.PublishNotificationRequest{
		NotificationType: DevsiteCompanyInvite,
		Recipient:        &receiver.PublishNotificationRequest_RecipientId{RecipientId: userId},
		Metadata:         CreateMetadataForEmail("organization_name", organizationName),
	}

	traceId, err := r.PublishNotification(ctx, dartRequest)
	if err != nil {
		return "", err
	}
	return traceId, nil
}

func (r dartImpl) SendDeveloperCompanyActionApproved(ctx context.Context, userId string, organizationName string) (string, error) {

	dartRequest := &receiver.PublishNotificationRequest{
		NotificationType: DeveloperCompanyActionApproved,
		Recipient:        &receiver.PublishNotificationRequest_RecipientId{RecipientId: userId},
		Metadata:         CreateMetadataForEmail("company_name", organizationName),
	}

	traceId, err := r.PublishNotification(ctx, dartRequest)
	if err != nil {
		return "", err
	}
	return traceId, nil
}

func (r dartImpl) SendDeveloperCompanyActionDenied(ctx context.Context, userId string, organizationName string) (string, error) {

	dartRequest := &receiver.PublishNotificationRequest{
		NotificationType: DeveloperCompanyActionDenied,
		Recipient:        &receiver.PublishNotificationRequest_RecipientId{RecipientId: userId},
		Metadata:         CreateMetadataForEmail("organization_name", organizationName),
	}

	traceId, err := r.PublishNotification(ctx, dartRequest)
	if err != nil {
		return "", err
	}
	return traceId, nil
}

func (r dartImpl) SendDeveloperCompanyUserActionAdded(ctx context.Context, userId string, organizationName string) (string, error) {

	dartRequest := &receiver.PublishNotificationRequest{
		NotificationType: DeveloperCompanyUserActionAdded,
		Recipient:        &receiver.PublishNotificationRequest_RecipientId{RecipientId: userId},
		Metadata:         CreateMetadataForEmail("company_name", organizationName),
	}

	traceId, err := r.PublishNotification(ctx, dartRequest)
	if err != nil {
		return "", err
	}
	return traceId, nil
}

func (r dartImpl) SendDeveloperCompanyUserActionRemoved(ctx context.Context, userId string, organizationName string) (string, error) {

	dartRequest := &receiver.PublishNotificationRequest{
		NotificationType: DeveloperCompanyUserActionRemoved,
		Recipient:        &receiver.PublishNotificationRequest_RecipientId{RecipientId: userId},
		Metadata:         CreateMetadataForEmail("company_name", organizationName),
	}

	traceId, err := r.PublishNotification(ctx, dartRequest)
	if err != nil {
		return "", err
	}
	return traceId, nil
}

func (r dartImpl) SendDeveloperCompanyGameActionApproved(ctx context.Context, userId string, gameName string) (string, error) {

	dartRequest := &receiver.PublishNotificationRequest{
		NotificationType: DeveloperCompanyGameActionApproved,
		Recipient:        &receiver.PublishNotificationRequest_RecipientId{RecipientId: userId},
		Metadata:         CreateMetadataForEmail("game_name", gameName),
	}

	traceId, err := r.PublishNotification(ctx, dartRequest)
	if err != nil {
		return "", err
	}
	return traceId, nil
}

func (r dartImpl) SendDeveloperCompanyGameActionDenied(ctx context.Context, userId string, gameName string) (string, error) {

	dartRequest := &receiver.PublishNotificationRequest{
		NotificationType: DeveloperCompanyGameActionDenied,
		Recipient:        &receiver.PublishNotificationRequest_RecipientId{RecipientId: userId},
		Metadata:         CreateMetadataForEmail("game_name", gameName),
	}

	traceId, err := r.PublishNotification(ctx, dartRequest)
	if err != nil {
		return "", err
	}
	return traceId, nil
}

func CreateMetadataForEmail(key string, value string) map[string]string {
	metadata := make(map[string]string)
	metadata[key] = value
	return metadata
}
