package extensions

import (
	"context"

	"code.justin.tv/foundation/twitchclient"
)

// Client is a simplified interface for the Extensions EMS system.
// The cartmanToken parameter is used as the Twith-Authorization header, it needs to have the appropriate capabilities.
// Errors may be wrapped twitchclient.Error, check the status with errorutil.StatusCode(err).
//go:generate errxer Client
type Client interface {

	// GetExtensionVersion returns an extension version.
	// Errors: 404: Not Found, 401 or 403: authorization issue.
	GetExtensionVersion(ctx context.Context, extensionID, version, cartmanToken string) (*ExtensionVersion, error)

	// Set state to "review", "approved" or "pending_action"
	// Errors: 404: Not Found, 401 or 403: authorization issue.
	TransitionExtensionState(ctx context.Context, extensionID, state, version, cartmanToken string) error
}

func NewClient(clientConf twitchclient.ClientConf) Client {
	httpCli := twitchclient.NewHTTPClient(clientConf)
	jsonCLi := twitchclient.NewJSONClient(clientConf.Host, httpCli)
	client := &client{JSONClient: jsonCLi}
	return &ClientErrx{Client: client}
}

type client struct {
	JSONClient *twitchclient.JSONClient
}

func (c *client) GetExtensionVersion(ctx context.Context, extensionID, version, cartmanToken string) (*ExtensionVersion, error) {
	path := twitchclient.Pathf("/extensions/%s/%s", extensionID, version)

	ctx = withReqOpts(ctx, cartmanToken, "get_extension_version")
	var respBody ExtensionVersion
	err := c.JSONClient.Get(ctx, path, nil, &respBody)
	return &respBody, err
}

func (c *client) TransitionExtensionState(ctx context.Context, extensionID, version, state, cartmanToken string) error {
	path := twitchclient.Pathf("/extensions/%s/%s", extensionID, version)
	body := stateTransition{State: state}

	ctx = withReqOpts(ctx, cartmanToken, "transition_extension_state")
	return c.JSONClient.Put(ctx, path, body, nil)
}

//
// Helpers
//

func withReqOpts(ctx context.Context, cartmanToken, statname string) context.Context {
	return twitchclient.WithReqOpts(ctx, twitchclient.ReqOpts{
		AuthorizationToken: cartmanToken,
		StatName:           "service.extensions." + statname,
		StatSampleRate:     1,
	})
}

//
// Models
//

type ExtensionVersion struct {
	ID          string `json:"id"`
	Version     string `json:"version"`
	Name        string `json:"name"`
	Description string `json:"description"`
	Summary     string `json:"summary"`
	AuthorName  string `json:"author_name"`

	// State should be "review", "approved", etc.
	// but the EMS endpoint is returning "In Review", etc.
	// This should be fixed later, using GetExtensionManifest EMS endpoint (PR: https://git-aws.internal.justin.tv/devrel/devsite-rbac/pull/178)
	State string `json:"state"`
}

type stateTransition struct {
	State string `json:"state"`
}
