package kmsclient

import (
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/kms"
)

//go:generate counterfeiter . KmsClient
type KmsClient interface {
	Decrypt(input *kms.DecryptInput) (*kms.DecryptOutput, error)
	GenerateDataKey() (*kms.GenerateDataKeyOutput, error)
	GetKeyARN() string
	GetRegion() string
}

type kmsClientImpl struct {
	client kms.KMS
	keyARN string
	region string
}

const (
	AES256GCM = "AES_256"
)

func NewClient(keyArn string, region string) (KmsClient, error) {
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String(region)},
	)

	if err != nil {
		return &kmsClientImpl{}, err
	}

	// Create KMS service client
	svc := kms.New(sess)
	return &kmsClientImpl{
		client: *svc,
		keyARN: keyArn,
		region: region,
	}, nil
}

func (c *kmsClientImpl) Decrypt(input *kms.DecryptInput) (*kms.DecryptOutput, error) {
	return c.client.Decrypt(input)
}

func (c *kmsClientImpl) GenerateDataKey() (*kms.GenerateDataKeyOutput, error) {
	return c.client.GenerateDataKey(&kms.GenerateDataKeyInput{
		KeySpec: aws.String(AES256GCM),
		KeyId:   aws.String(c.keyARN),
	})
}

func (c *kmsClientImpl) GetKeyARN() string {
	return c.keyARN
}

func (c *kmsClientImpl) GetRegion() string {
	return c.region
}
