package owlcli

import (
	"context"
	"net/url"

	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/foundation/twitchclient"
	owl "code.justin.tv/web/owl/client"
	"code.justin.tv/web/owl/oauth2"
)

// Client is a simplified interface for the Owl Service
//go:generate errxer Client
type Client interface {
	// Validate OAuth token with empty scopes
	// The response has the Valid and Exp to check for validation and expiration.
	// The error could be one of owl errors: owl.ErrInvalidClientID, owl.ErrForbiddenClientID, etc.
	Validate(ctx context.Context, token string) (*oauth2.Authorization, error)

	// Get an Owl Client
	GetClient(ctx context.Context, clientID string) (*oauth2.Client, error)

	// Update client ownerID (user-twitchID) and groupID (companyID).
	UpdateClientOwner(ctx context.Context, params UpdateClientRequest) error
}

type owlImpl struct {
	client owl.Client
}

func NewClient(clientConf twitchclient.ClientConf) (Client, error) {
	client, err := owl.NewClient(clientConf)
	if err != nil {
		return nil, err
	}
	return &ClientErrx{Client: &owlImpl{client: client}}, nil
}

func (o *owlImpl) Validate(ctx context.Context, token string) (*oauth2.Authorization, error) {
	return o.client.Validate(ctx, token, nil, defaultReqOpts())
}

func (o *owlImpl) GetClient(ctx context.Context, clientID string) (*oauth2.Client, error) {
	return o.client.GetClient(ctx, clientID, defaultReqOpts())
}

type UpdateClientRequest struct {
	ClientID           string
	OwnerID            string
	GroupID            string
	RequestingTwitchID string
}

const (
	// ResetGroupID will set owl client's group_id to null.
	ResetGroupID = "NULL"
)

func (o *owlImpl) UpdateClientOwner(ctx context.Context, params UpdateClientRequest) error {
	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"Owl.ClientID", params.ClientID},
		{"Owl.GroupID", params.GroupID},
		{"Owl.RequestingTwitchID", params.RequestingTwitchID},
	}); err != nil {
		return err
	}

	if params.GroupID == ResetGroupID {
		params.GroupID = "" // owl uses empty string to set value to null
	}

	reqOwnerID := &params.OwnerID
	if params.OwnerID == "" {
		// do not change ownerID
		reqOwnerID = nil
	}

	q := url.Values{"user_id": {params.RequestingTwitchID}}
	_, err := o.client.UpdateClient(ctx, params.ClientID, owl.UpdateClientRequest{
		OwnerID: reqOwnerID,
		GroupID: &params.GroupID,
	}, q, defaultReqOpts())
	return err
}

func defaultReqOpts() *twitchclient.ReqOpts {
	return &twitchclient.ReqOpts{
		StatSampleRate: 1,
	}
}
