package passport

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"

	"code.justin.tv/chat/golibs/logx"

	"code.justin.tv/foundation/twitchclient"
	identitypassport "code.justin.tv/identity/passport/client"
	"code.justin.tv/identity/passport/passport/errors"
)

type CreateNewAccountRequest struct {
	ClientID string   `json:"client_id,omitempty"`
	Username string   `json:"username,omitempty"`
	Email    string   `json:"email,omitempty"`
	Birthday Birthday `json:"birthday,omitempty"`
}

type CreateNewAccountResponse struct {
	AccessToken      string `json:"access_token,omitempty"`
	ErrorTranslated  string `json:"error,omitempty"`
	ErrorCode        int    `json:"error_code,omitempty"`
	ErrorDescription string `json:"error_description,omitempty"`
}

type Birthday struct {
	Day   int `json:"day"`
	Month int `json:"month"`
	Year  int `json:"year"`
}

//go:generate errxer Client
//go:generate counterfeiter . Client
type Client interface {
	GetTwoFactorEnabled(ctx context.Context, twitchID string) (bool, error)
	SignUpNewAccount(ctx context.Context, req CreateNewAccountRequest) (CreateNewAccountResponse, error)
}

type client struct {
	passportClient identitypassport.Client
	httpClient     *http.Client
	host           string
}

func NewClient(conf twitchclient.ClientConf) (Client, error) {
	c, err := identitypassport.NewClient(conf)
	if err != nil {
		return nil, err
	}

	return &client{
		passportClient: c,
		httpClient:     twitchclient.NewHTTPClient(conf),
		host:           conf.Host,
	}, nil
}

func (c *client) GetTwoFactorEnabled(ctx context.Context, twitchID string) (bool, error) {
	enabled, err := c.passportClient.GetTwoFactorEnabled(ctx, twitchID, nil)
	logx.Info(ctx, "user's two factor status", logx.Fields{
		"twitch_id": twitchID,
		"2fa_state": enabled,
	})
	return enabled, err
}

func (c *client) SignUpNewAccount(ctx context.Context, req CreateNewAccountRequest) (CreateNewAccountResponse, error) {
	jsonBytes, err := json.Marshal(&req)
	if err != nil {
		return CreateNewAccountResponse{}, errors.InvalidJSON
	}

	// trusted_request is for bypassing the rate limiting:
	// https://git.xarth.tv/identity/passport#bypassing-new-client-verification-ncv-captcha-and-rate-limits
	request, err := http.NewRequest(http.MethodPost, c.host+"/api/v1/internal/register?shadow=true&trusted_request=true", bytes.NewBuffer(jsonBytes))
	if err != nil {
		return CreateNewAccountResponse{}, err
	}

	resp, err := c.httpClient.Do(request)
	if err != nil {
		return CreateNewAccountResponse{}, err
	}

	defer resp.Body.Close()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return CreateNewAccountResponse{}, err
	}

	var respEntity CreateNewAccountResponse
	err = json.Unmarshal(body, &respEntity)
	if err != nil {
		return CreateNewAccountResponse{}, err
	}

	if len(respEntity.ErrorTranslated) > 0 {
		newErr := fmt.Errorf("error: %s, error_code: %d, message: %s", respEntity.ErrorTranslated, respEntity.ErrorCode, respEntity.ErrorDescription)
		logx.Error(ctx, newErr, logx.Fields{
			"error_description": respEntity.ErrorDescription,
			"error_code":        respEntity.ErrorCode,
			"errors":            respEntity.ErrorTranslated,
			"username":          req.Username,
			"Email":             req.Email,
		})
		return CreateNewAccountResponse{}, newErr
	}

	if respEntity.AccessToken == "" {
		newErr := errors.New("failed to sign up new account - no access_token returned. please try again later")
		logx.Error(ctx, newErr, logx.Fields{
			"error_description": "no access_token returned, please try again later",
			"username":          req.Username,
			"Email":             req.Email,
		})
		return CreateNewAccountResponse{}, newErr
	}

	return respEntity, nil
}
