package salesforce

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/url"
	"time"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/devrel/devsite-rbac/internal/jwt"
)

const (
	// Our JWT tokens should be used immediately by Salesforce
	jwtExpiresIn = 5 * time.Minute

	// Salesforce tokens are set to expire in 2 hours, it is good for us to refresh every 1 hour.
	accessTokenExpiresIn = 1 * time.Hour
)

// ensureAuthenticated checks if the last loaded AccessToken is not expired,
// if expired or not loaded yet, then calls the oauth2/token API to generate a new token.
func (c *client) ensureAuthenticated(ctx context.Context) error {
	// Check the current AccessToken is still fresh
	if c.AccessTokenExpires.After(c.Now()) {
		return nil
	}

	resp, err := c.getOAuth2Token(ctx)
	if err != nil {
		return errx.Wrap(err, "ensureAuthenticated")
	}

	c.InstanceURL = resp.InstanceURL
	c.AccessToken = resp.AccessToken
	c.AccessTokenExpires = c.Now().Add(accessTokenExpiresIn)
	return nil
}

// getOAuth2Token calls the Salesforce API /oauth2/token to get a new token.
// It uses JWT authentication, we have a private key (in Secrets Manager) and Salesforce
// has the public key, so they can verify that we are the ones sending the request. See
// https://salesforce.stackexchange.com/questions/204994/connected-app-use-external-clientid-in-jwt
// And here it is a good article about JWT and RS256 signing algorithm:
// https://blog.miguelgrinberg.com/post/json-web-tokens-with-public-key-signatures
func (c *client) getOAuth2Token(ctx context.Context) (*oauth2TokenResp, error) {
	// Encode a new JWT
	key := c.Config.RSA256PrivKey
	claims := jwt.Claims{
		Audience: c.Config.Host,
		Expires:  c.Now().Add(jwtExpiresIn).Unix(),
		Issuer:   c.Config.ClientID,
		Subject:  c.Config.Username,
	}
	jwtBytes, err := jwt.EncodeRS256([]byte(key), claims)
	if err != nil {
		return nil, errx.Wrap(err, "oauth: failed to do jwt.EncodeRS256")
	}
	jwt := string(jwtBytes)

	// Request token in the API
	apiURL := c.Config.Host + "/services/oauth2/token?" + url.Values{
		"assertion":  {jwt},
		"grant_type": {"urn:ietf:params:oauth:grant-type:jwt-bearer"},
	}.Encode()
	resp, err := c.doRequest(ctx, "getOAuth2Token", "POST", apiURL, nil)
	defer closeBody(ctx, resp)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != 200 {
		body, _ := ioutil.ReadAll(resp.Body)
		return nil, fmt.Errorf("Salesforce /oauth2/token: response status %d: %s", resp.StatusCode, string(body))
	}

	respData := oauth2TokenResp{}
	err = json.NewDecoder(resp.Body).Decode(&respData)
	return &respData, err
}

func (c *client) Now() time.Time {
	return time.Now() // encapsulate here because in the future we may want to mock the current time on tests
}

type oauth2TokenResp struct {
	AccessToken string `json:"access_token"`
	InstanceURL string `json:"instance_url"`
	ID          string `json:"id"`
	Scope       string `json:"scope"`
	TokenType   string `json:"token_type"`
}
