package salesforce

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"time"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
)

type Config struct {
	Host          string // URL to authenticate (<Host>/services/oauth2/token) and used as Audience in the JWT claims
	Username      string // used as Subject in the JWT claims
	ClientID      string // used as Issuer in the JWT claims, represents Vienna in Salesforce
	RSA256PrivKey string // to sign the JWT token (Salesforce has the public key to verify)

	Stats statter
}

type statter interface {
	TimingDuration(stat string, delta time.Duration, rate float32) error
}

// Client is a Salesforce client, used to integrate Vienna/RBAC with Salesforce.
// It uses JWT authentication, which means Salesforce needs to be configured to receive requests from the Vienna client,
// using a certificate that was made from the public key, so they can verify that we sent the requests.
// NOTE: staging is configured to use a Salesforce Sandbox, which is never updated. When new functionality is
// implemented, the Salesforce consultantes will probably need to make a new Sandbox.
//go:generate errxer Client
type Client interface {

	// Create a new Case for an Extension Review
	CreateExtensionReviewCase(ctx context.Context, extCase ExtensionReviewCase) (salesforceID string, err error)
}

type client struct {
	Config     Config
	HTTPClient httpClientDoer

	InstanceURL        string    // set after authenticate
	AccessToken        string    // set after authenticate
	AccessTokenExpires time.Time // after this time, the client should re-authenticate
}

type httpClientDoer interface {
	Do(req *http.Request) (*http.Response, error)
}

const httpTimeout = 30 * time.Second

func NewSalesforceClient(config Config) (Client, error) {
	client := &client{
		Config:     config,
		HTTPClient: &http.Client{Timeout: httpTimeout},
	}
	err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"Config.Host", config.Host},
		{"Config.Username", config.Username},
		{"Config.ClientID", config.ClientID},
		{"Config.RSA256PrivKey", config.RSA256PrivKey},
	})
	return &ClientErrx{Client: client}, err
}

func (c *client) CreateExtensionReviewCase(ctx context.Context, extCase ExtensionReviewCase) (string, error) {
	if err := validateExtensionReviewCase(extCase); err != nil {
		return "", err
	}
	if err := c.ensureAuthenticated(ctx); err != nil {
		return "", err
	}

	// https://developer.salesforce.com/docs/api-explorer/sobject/Case/post-case
	apiURL := fmt.Sprintf("%s/services/data/v43.0/sobjects/Case/", c.InstanceURL)
	resp, err := c.doRequest(ctx, "CreateExtensionReviewCase", "POST", apiURL, extCase)
	defer closeBody(ctx, resp)
	if err != nil {
		return "", err
	}

	if resp.StatusCode != 201 { // Failed to create
		var respStr string
		bodyBytes, err := ioutil.ReadAll(resp.Body)
		if err == nil {
			respStr = string(bodyBytes)
		} else {
			respStr = "Failed to read body: " + err.Error()
		}
		return "", fmt.Errorf("Salesforce API: CreateExtensionReviewCase: response status %d: %s", resp.StatusCode, respStr)
	}

	respData := apiCreateResp{}
	err = json.NewDecoder(resp.Body).Decode(&respData)
	if err != nil {
		return "", err
	}
	if !respData.Success {
		return "", fmt.Errorf("Salesforce API: CreateExtensionReviewCase: response success: false, errors: %v", respData.Errors)
	}

	return respData.ID, nil
}

//
// Helpers
//

func (c *client) doRequest(ctx context.Context, statname, method, url string, jsonData interface{}) (resp *http.Response, err error) {
	// Track Stats for duration depending on the response status code
	status := 0
	start := time.Now()
	defer func() {
		if c.Config.Stats == nil {
			return
		}
		duration := time.Since(start)
		if resp != nil {
			status = resp.StatusCode
		}
		stat := fmt.Sprintf("salesforceAPI.doRequest.%s.%d", statname, status)
		_ = c.Config.Stats.TimingDuration(stat, duration, 1.0)
	}()

	// Serialize request body as JSON
	var body io.Reader
	if jsonData != nil {
		jsonBytes, err := json.Marshal(jsonData)
		if err != nil {
			return nil, err
		}
		body = bytes.NewBuffer(jsonBytes)
	}

	// New request
	req, err := http.NewRequest(method, url, body)
	if err != nil {
		return nil, err
	}
	req = req.WithContext(ctx)

	// Add headers
	if jsonData != nil {
		req.Header.Set("Content-Type", "application/json")
	}
	if c.AccessToken != "" {
		req.Header.Set("Authorization", fmt.Sprintf("Bearer %s", c.AccessToken))
	}

	// Do request
	return c.HTTPClient.Do(req)
}

func closeBody(ctx context.Context, resp *http.Response) {
	if resp == nil || resp.Body == nil {
		return
	}
	err := resp.Body.Close()
	if err != nil {
		logx.Error(ctx, errx.New(err))
	}
}

func validateExtensionReviewCase(extCase ExtensionReviewCase) error {
	return errorutil.ValidateRequiredArgs(errorutil.Args{
		{"ExtensionReviewCase.Subject", extCase.Subject},
		{"ExtensionReviewCase.Origin", extCase.Origin},
		{"ExtensionReviewCase.Description", extCase.Description},
		{"ExtensionReviewCase.ContactEmail", extCase.ContactEmail},
		{"ExtensionReviewCase.SuppliedEmail", extCase.SuppliedEmail},
		{"ExtensionReviewCase.TwitchUserID", extCase.TwitchUserID},
		{"ExtensionReviewCase.TwitchLogin", extCase.TwitchLogin},
		{"ExtensionReviewCase.ExtensionID", extCase.ExtensionID},
		{"ExtensionReviewCase.ExtensionVersion", extCase.ExtensionVersion},
		{"ExtensionReviewCase.TwitchTestChannel", extCase.TwitchTestChannel},
	})
}

//
// Models
//

type ExtensionReviewCase struct {
	Subject       string `json:"Subject"`       // Extension Name
	Origin        string `json:"Origin"`        // always "Vienna"
	Type          string `json:"Type"`          // always "Extension"
	RecordTypeID  string `json:"RecordTypeId"`  // always "0121K000000vgGWQAY", DX Review report type, denotes layouts and such
	Description   string `json:"Description"`   // release notes added by the developer
	SuppliedEmail string `json:"SuppliedEmail"` // Extension author email

	TwitchUserID      string `json:"Developer_Id__c"`   // developer twitch id
	TwitchLogin       string `json:"TwitchUsername__c"` // developer twitch login
	ContactEmail      string `json:"Twitch_Email__c"`   // developer twitch email
	ExtensionID       string `json:"Extension_Id__c"`
	ExtensionVersion  string `json:"Extension_Version__c"`
	TwitchTestChannel string `json:"Twitch_Test_Channel__c"` // channel URL to test the extension
}

type apiCreateResp struct {
	ID      string   `json:"id"`
	Errors  []string `json:"errors"`
	Success bool     `json:"success"`
}
