package users

import (
	"context"
	"strconv"

	"code.justin.tv/foundation/twitchclient"
	clientutils "code.justin.tv/web/users-service/client"
	"code.justin.tv/web/users-service/client/usersclient_internal"
	"code.justin.tv/web/users-service/models"
)

// Users is a simplified interface for the Users Service (https://git-aws.internal.justin.tv/web/users-service)
//go:generate errxer Users
//go:generate counterfeiter . Users
type Users interface {
	GetUserByID(ctx context.Context, twitchID string) (*models.Properties, error)
	GetUserByLogin(ctx context.Context, login string) (*models.Properties, error)
	IsValidTwitchID(ctx context.Context, twitchID string) (bool, error)
}

type usersImpl struct {
	client usersclient_internal.InternalClient
}

func NewUsers(clientConf twitchclient.ClientConf) (Users, error) {
	client, err := usersclient_internal.NewClient(clientConf)
	if err != nil {
		return nil, err
	}
	return &UsersErrx{Users: &usersImpl{client: client}}, nil
}

func (u *usersImpl) GetUserByID(ctx context.Context, twitchID string) (*models.Properties, error) {
	return u.client.GetUserByID(ctx, twitchID, defaultReqOpts())
}

func (u *usersImpl) GetUserByLogin(ctx context.Context, login string) (*models.Properties, error) {
	return u.client.GetUserByLogin(ctx, login, defaultReqOpts())
}

func (u *usersImpl) IsValidTwitchID(ctx context.Context, twitchID string) (bool, error) {
	if _, err := strconv.Atoi(twitchID); err != nil {
		return false, nil // bad format, the id is supposed to be numeric
	}

	_, err := u.client.GetUserByID(ctx, twitchID, defaultReqOpts())
	if clientutils.IsUserNotFound(err) {
		return false, nil // user does not exist
	}
	if err != nil {
		return false, err // oops, internal error, we don't know if the users exists or not
	}

	return true, nil // all good, user exists therefor the id is valid
}

func defaultReqOpts() *twitchclient.ReqOpts {
	return &twitchclient.ReqOpts{
		StatSampleRate: 1,
	}
}
