package config_test

import (
	"errors"
	"testing"

	"code.justin.tv/devrel/devsite-rbac/config"
	"github.com/stretchr/testify/require"
)

func TestLoad_Success(t *testing.T) {
	// Use e2e configuration because it doesn't need secrets and the values are quite stable for testing.
	e2eConf, err := config.Load("./env-e2e.toml")
	require.NoError(t, err)
	require.Equal(t, "NOOP", e2eConf.StatsdHostPort)
	require.Equal(t, "changeme", e2eConf.PGPassword)
}

func TestLoad_InvalidFilePath(t *testing.T) {
	_, err := config.Load("./env-nowheretobefound.toml")
	require.EqualError(t, err, "open ./env-nowheretobefound.toml: no such file or directory")
}

func TestLoad_InvalidTOMLFormat(t *testing.T) {
	_, err := config.Load("./config.go") // Use a .go file, which is clearly not valid TOML
	// the error message should be stable, because the go file always starts with the package declaration.
	require.EqualError(t, err, "(1, 9): was expecting token =, but got keys cannot contain new lines instead")
}

func TestSetValues(t *testing.T) {
	c := &config.RBAC{
		PGHost:       "unit_test",
		PGSSLEnabled: true,
	}
	err := c.Set("PGHost", "updated")
	require.NoError(t, err)
	require.Equal(t, "updated", c.PGHost)

	err = c.Set("PGHost", "")
	require.NoError(t, err)
	require.Equal(t, "", c.PGHost)

	require.Equal(t, true, c.PGSSLEnabled)
	err = c.Set("PGSSLEnabled", false)
	require.NoError(t, err)
	require.Equal(t, false, c.PGSSLEnabled)

	err = c.Set("Unexsiting-field", "foo")
	require.EqualError(t, err, "conf.Set(Unexsiting-field): invalid key")

	err = c.Set("PGPort", 333) // invalid type
	require.EqualError(t, err, "conf.Set(PGPort): expected type string, found int")

	err = c.Set("EXTENDS", "")
	require.NoError(t, err)
}

func TestUpdateMatchingFields_Success(t *testing.T) {
	c := &config.RBAC{
		Environment: "unit_test",
		PGHost:      "{{testprefix:foobarval}}",
		PGPort:      "666",
		PGUser:      "{{otherprefix:leave me alone}}",
	}

	err := c.UpdateMatchingFields("{{testprefix:", "}}", func(fieldName string, trimmed string) (string, error) {
		return fieldName + "-" + trimmed + "-it works", nil
	})

	require.NoError(t, err)
	require.Equal(t, "unit_test", c.Environment)
	require.Equal(t, "PGHost-foobarval-it works", c.PGHost) // UPDATED
	require.Equal(t, "666", c.PGPort)
	require.Equal(t, "{{otherprefix:leave me alone}}", c.PGUser) // Not updated, does not match the prefix
}

func TestUpdateMatchingFields_Error(t *testing.T) {
	c := &config.RBAC{
		Environment: "unit_test",
		PGHost:      "prefix::please fail me",
	}

	err := c.UpdateMatchingFields("prefix::", "", func(fieldName string, trimmed string) (string, error) {
		if trimmed == "please fail me" {
			return trimmed, errors.New("As you please")
		}
		return trimmed, nil
	})

	require.EqualError(t, err, "As you please")
	require.Equal(t, "unit_test", c.Environment)
	require.Equal(t, "prefix::please fail me", c.PGHost) // Not updated, because there was an error
}
