package e2e

import (
	"context"
	"strings"
	"testing"

	"github.com/twitchtv/twirp"

	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/stretchr/testify/suite"
)

func Test_CompanySuite(t *testing.T) {
	s := &CompanySuite{}
	suite.Run(t, s)
}

type CompanySuite struct {
	Suite
}

func (s *CompanySuite) Test_CreateCompany() {
	company := s.createCompany()

	companyResp, err := s.RBAC.GetCompanyApplication(s.Ctx, &rbacrpc.Id{
		Id: company.Id,
	})
	s.Nil(err)
	s.NotNil(companyResp)
	s.Equal(companyResp.Id, company.Id)
}

func (s *CompanySuite) Test_CreateCompany_OwnerUserCreated() {
	company := s.createCompany()

	// Get admin created with company
	usersResp, err := s.RBAC.GetUsersByCompanyId(s.Ctx, &rbacrpc.GetUsersByCompanyIdRequest{Id: company.Id})
	s.NoError(err)

	counts := countMemberships(usersResp.Memberships)
	s.Equal(counts.Owners, 1)
	s.Equal(counts.Admins, 0)
	s.Equal(counts.Managers, 0)
	s.Equal(counts.Marketers, 0)
	s.Equal(counts.Developers, 0)
}

func (s *CompanySuite) Test_AddUsersToCompany_ForAllRoles() {
	company := s.createCompany()
	s.createMembershipsForAllRoles(company.Id)

	counts := s.membershipRoleCounts(company.Id)
	s.Equal(counts.Owners, 1)
	s.Equal(counts.Admins, 1)
	s.Equal(counts.Managers, 1)
	s.Equal(counts.Marketers, 1)
	s.Equal(counts.Developers, 1)
}

func (s *CompanySuite) Test_AddUsersToCompany_WithLowercaseRoles() {
	company := s.createCompany()

	resp, err := s.RBAC.ListAllRoles(s.Ctx, &rbacrpc.Empty{})
	s.NoError(err)
	for _, role := range resp.Roles {
		if role == "Owner" {
			continue
		}

		s.mustCreateMembership(membershipParams{
			CompanyID: company.Id,
			Role:      strings.ToLower(role),
		})
	}

	counts := s.membershipRoleCounts(company.Id)
	s.Equal(counts.Owners, 1)
	s.Equal(counts.Admins, 1)
	s.Equal(counts.Managers, 1)
	s.Equal(counts.Marketers, 1)
	s.Equal(counts.Developers, 1)

	developer := s.membershipWithRole(company.Id, "Developer")

	// Update role using lowercase notation
	_, err = s.RBAC.UpdateUserRole(s.AdminCtx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:  developer.TwitchId,
		Role:      "developer", // no change but shouldn't fail
		CompanyId: company.Id,
	})
	s.NoError(err)
}

func (s *CompanySuite) Test_AddUsersToCompany_ActivityLogs() {
	company := s.createCompany()
	numRolesCreated := s.createMembershipsForAllRoles(company.Id)

	checkCreatedUserActivityLogs := func(users []*rbacrpc.Membership) {
		for _, user := range users {
			s.checkAuthedEntityActionsLen(user.TwitchId, "UserRole", WhitelistAdminTwitchId, company.Id, 1)
		}
	}

	checkCreatedUserActivityLogs(s.membershipsWithRole(company.Id, "Administrator"))
	checkCreatedUserActivityLogs(s.membershipsWithRole(company.Id, "Manager"))
	checkCreatedUserActivityLogs(s.membershipsWithRole(company.Id, "Marketer"))
	checkCreatedUserActivityLogs(s.membershipsWithRole(company.Id, "Developer"))
	s.checkAuthedCompanyActionsLen(company.Id, WhitelistAdminTwitchId, numRolesCreated+7)
}

func (s *CompanySuite) Test_AddUsersToCompany_AlreadyExists() {
	company := s.createCompany()
	owner := s.membershipWithRole(company.Id, "Owner")

	manager := s.mustCreateMembership(membershipParams{
		CompanyID:          company.Id,
		Role:               "Manager",
		RequestingTwitchID: owner.TwitchId,
	})

	// Adding the same member again returns already_exists error
	_, err := s.createMembership(membershipParams{
		CompanyID:          company.Id,
		Role:               "Manager",
		TwitchID:           manager.TwitchId, // same user
		RequestingTwitchID: owner.TwitchId,
	})
	s.EqualErrorCode(err, twirp.AlreadyExists)
}

func (s *CompanySuite) Test_AddUserToCompany_AsRandomUserNotAllowed() {
	company := s.createCompany()

	// Other random users from other companies are not allowed
	_, err := s.createMembership(membershipParams{
		CompanyID:          company.Id,
		Role:               "Developer",
		RequestingTwitchID: randomTwitchID(),
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)
}

func (s *CompanySuite) Test_AddUserToCompany_AsDeveloperNotAllowed() {
	company := s.createCompany()

	devUser := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Developer",
	})

	// Dev user is not allowed to add more users
	_, err := s.createMembership(membershipParams{
		CompanyID:          company.Id,
		Role:               "Developer",
		RequestingTwitchID: devUser.TwitchId,
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)
}

func (s *CompanySuite) Test_AddUserToCompany_AsManager() {
	company := s.createCompany()

	managerUser := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Manager",
	})

	// Manager user is not allowed to add administrators
	_, err := s.createMembership(membershipParams{
		CompanyID:          company.Id,
		Role:               "Administrator",
		RequestingTwitchID: managerUser.TwitchId,
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)

	// Manager user is not allowed to add owners
	_, err = s.createMembership(membershipParams{
		CompanyID:          company.Id,
		Role:               "Owner",
		RequestingTwitchID: managerUser.TwitchId,
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)

	// Manager user can add managers
	m, err := s.createMembership(membershipParams{
		CompanyID:          company.Id,
		Role:               "Manager",
		RequestingTwitchID: managerUser.TwitchId,
	})
	s.NoError(err)
	s.Equal("Manager", m.Role)

	// Manager user can add developers
	m, err = s.createMembership(membershipParams{
		CompanyID:          company.Id,
		Role:               "Developer",
		RequestingTwitchID: managerUser.TwitchId,
	})
	s.NoError(err)
	s.Equal("Developer", m.Role)

	// Manager user can add marketers
	m, err = s.createMembership(membershipParams{
		CompanyID:          company.Id,
		Role:               "Marketer",
		RequestingTwitchID: managerUser.TwitchId,
	})
	s.NoError(err)
	s.Equal("Marketer", m.Role)
}

func (s *CompanySuite) Test_AddUserToCompany_BillingManager() {
	company := s.createCompany()
	owner := s.membershipWithRole(company.Id, "Owner")

	// User didn't complete TIMS
	_, err := s.createMembership(membershipParams{
		CompanyID:          company.Id,
		Role:               "Billing_Manager",
		TwitchID:           randomTwitchID(),
		RequestingTwitchID: owner.TwitchId,
	})
	s.EqualErrorCode(err, twirp.FailedPrecondition)

	// User with TIMs completed
	m, err := s.createMembership(membershipParams{
		CompanyID:          company.Id,
		Role:               "Billing_Manager",
		TwitchID:           randomTwitchIDWithTIMsEnabled(),
		RequestingTwitchID: owner.TwitchId,
	})
	s.NoError(err)
	s.Equal("Billing_Manager", m.Role)

	mReload := s.membershipWithRole(company.Id, "Billing_Manager")
	s.Equal(m.TwitchId, mReload.TwitchId)
}

func (s *CompanySuite) Test_GetMembership() {
	company := s.createCompany()
	owner := s.membershipWithRole(company.Id, "Owner")

	// Success
	memb, err := s.RBAC.GetMembership(s.Ctx, &rbacrpc.GetMembershipRequest{
		RequestingTwitchId: owner.TwitchId,
		TwitchId:           owner.TwitchId,
		CompanyId:          company.Id,
	})
	s.NoError(err)
	s.Equal(owner.Role, memb.Role)
	s.Equal(owner.FirstName, memb.FirstName)

	// Not Found
	_, err = s.RBAC.GetMembership(s.Ctx, &rbacrpc.GetMembershipRequest{
		RequestingTwitchId: owner.TwitchId,
		TwitchId:           "idonotexist",
		CompanyId:          company.Id,
	})
	s.EqualErrorCode(err, twirp.NotFound)
}

func (s *CompanySuite) Test_UpdateUserRole() {
	company := s.createCompany()

	ownerUser := s.membershipWithRole(company.Id, "Owner")
	managerUser := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Manager",
	})
	adminUser := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Administrator",
	})
	newUser := s.mustCreateMembership(membershipParams{
		CompanyID:          company.Id,
		Role:               "Developer",
		RequestingTwitchID: adminUser.TwitchId,
	})

	// random user cannot update user
	_, err := s.RBAC.UpdateUserRole(s.Ctx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:           newUser.TwitchId,
		CompanyId:          company.Id,
		Role:               "Marketer",
		RequestingTwitchId: randomTwitchID(),
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)

	// random user cannot be updated
	_, err = s.RBAC.UpdateUserRole(s.Ctx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:           randomTwitchID(),
		CompanyId:          company.Id,
		Role:               "Marketer",
		RequestingTwitchId: adminUser.TwitchId,
	})
	s.EqualErrorCode(err, twirp.NotFound)

	// can not update owner's role
	for _, membership := range []*rbacrpc.Membership{
		adminUser,
		ownerUser,
		newUser,
	} {
		_, err = s.RBAC.UpdateUserRole(s.Ctx, &rbacrpc.UpdateUserRoleRequest{
			TwitchId:           ownerUser.TwitchId,
			CompanyId:          company.Id,
			Role:               "Developer",
			RequestingTwitchId: membership.TwitchId,
		})
		s.EqualErrorCode(err, twirp.PermissionDenied, membership.Role)
	}

	// can not update role to owner
	_, err = s.RBAC.UpdateUserRole(s.Ctx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:           newUser.TwitchId,
		CompanyId:          company.Id,
		Role:               "Owner",
		RequestingTwitchId: adminUser.TwitchId,
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)

	// Manager can not update user role to administrator
	_, err = s.RBAC.UpdateUserRole(s.Ctx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:           newUser.TwitchId,
		CompanyId:          company.Id,
		Role:               "Administrator",
		RequestingTwitchId: managerUser.TwitchId,
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)

	// happy path
	updMemb, err := s.RBAC.UpdateUserRole(s.Ctx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:           newUser.TwitchId,
		CompanyId:          company.Id,
		Role:               "Marketer",
		RequestingTwitchId: adminUser.TwitchId,
	})
	s.NoError(err)

	s.Equal(newUser.TwitchId, updMemb.TwitchId)
	s.Equal("Marketer", updMemb.Role) // returns updated value
}

func (s *CompanySuite) Test_UpdateUserRole_AsManager() {
	company := s.createCompany()

	managerUser := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Manager",
	})
	adminUser := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Administrator",
	})
	newUser := s.mustCreateMembership(membershipParams{
		CompanyID:          company.Id,
		Role:               "Developer",
		RequestingTwitchID: adminUser.TwitchId,
	})

	// Manager user is not allowed to update user role to administrator
	_, err := s.RBAC.UpdateUserRole(s.Ctx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:           newUser.TwitchId,
		CompanyId:          company.Id,
		Role:               "Administrator",
		RequestingTwitchId: managerUser.TwitchId,
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)

	// Manager user is not allowed to update user role to owner
	_, err = s.RBAC.UpdateUserRole(s.Ctx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:           newUser.TwitchId,
		CompanyId:          company.Id,
		Role:               "Owner",
		RequestingTwitchId: managerUser.TwitchId,
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)

	// Manager user can update user role to manager
	m, err := s.RBAC.UpdateUserRole(s.Ctx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:           newUser.TwitchId,
		CompanyId:          company.Id,
		Role:               "Manager",
		RequestingTwitchId: managerUser.TwitchId,
	})
	s.NoError(err)
	s.Equal("Manager", m.Role)

	// Manager user can update user role to developer
	m, err = s.RBAC.UpdateUserRole(s.Ctx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:           newUser.TwitchId,
		CompanyId:          company.Id,
		Role:               "Developer",
		RequestingTwitchId: managerUser.TwitchId,
	})
	s.NoError(err)
	s.Equal("Developer", m.Role)

	// Manager user can update user role to marketer
	m, err = s.RBAC.UpdateUserRole(s.Ctx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:           newUser.TwitchId,
		CompanyId:          company.Id,
		Role:               "Marketer",
		RequestingTwitchId: managerUser.TwitchId,
	})
	s.NoError(err)
	s.Equal("Marketer", m.Role)
}

func (s *CompanySuite) Test_UpdateUserRole_BillingManager() {
	company := s.createCompany()
	owner := s.membershipWithRole(company.Id, "Owner")

	userNoPayments := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Developer",
		TwitchID:  randomTwitchID(),
	})

	userWithTIMSEnabled := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Developer",
		TwitchID:  randomTwitchIDWithTIMsEnabled(),
	})

	// Can not assign Billing Manager role if user didn't complete TIMS
	_, err := s.RBAC.UpdateUserRole(s.Ctx, &rbacrpc.UpdateUserRoleRequest{
		CompanyId:          company.Id,
		Role:               "Billing_Manager",
		TwitchId:           userNoPayments.TwitchId,
		RequestingTwitchId: owner.TwitchId,
	})
	s.EqualErrorCode(err, twirp.FailedPrecondition)

	// User with TIMs completed is gucci
	_, err = s.RBAC.UpdateUserRole(s.Ctx, &rbacrpc.UpdateUserRoleRequest{
		CompanyId:          company.Id,
		Role:               "Billing_Manager",
		TwitchId:           userWithTIMSEnabled.TwitchId,
		RequestingTwitchId: owner.TwitchId,
	})
	s.NoError(err)

	mReload := s.membershipWithRole(company.Id, "Billing_Manager")
	s.Equal(userWithTIMSEnabled.TwitchId, mReload.TwitchId)
}

func (s *CompanySuite) Test_UpdateUserRole_BillingManager_WithExtension() {
	company := s.createCompany()
	owner := s.membershipWithRole(company.Id, "Owner")

	billingManager := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Billing_Manager",
		TwitchID:  randomTwitchIDWithTIMsEnabled(),
	})

	extensionID := "valid-client-id-5" // Client-ID assumed to be valid by the fake Owl client used in this e2e test
	_, err := s.RBAC.CreateResource(s.Ctx, &rbacrpc.CreateResourceRequest{
		CompanyId: company.Id,
		Resource: &rbacrpc.Resource{
			ExternalId: extensionID,
			Type:       "extension",
		},
	})
	s.NoError(err)

	_, err = s.RBAC.SetExtensionBillingManager(s.Ctx, &rbacrpc.SetExtensionBillingManagerRequest{
		ExtensionClientId:      extensionID,
		BillingManagerTwitchId: billingManager.TwitchId,
		RequestingTwitchId:     owner.TwitchId,
	})
	s.NoError(err)

	// user is assigned billing manager, cannot be removed
	_, err = s.RBAC.UpdateUserRole(s.Ctx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:           billingManager.TwitchId,
		CompanyId:          company.Id,
		Role:               "Developer",
		RequestingTwitchId: owner.TwitchId,
	})
	s.Error(err)

	anotherBillingManager := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Billing_Manager",
		TwitchID:  randomTwitchIDWithTIMsEnabled(),
	})

	// assign extension to another user
	_, err = s.RBAC.SetExtensionBillingManager(s.Ctx, &rbacrpc.SetExtensionBillingManagerRequest{
		ExtensionClientId:      extensionID,
		BillingManagerTwitchId: anotherBillingManager.TwitchId,
		RequestingTwitchId:     owner.TwitchId,
	})
	s.NoError(err)

	// can be updated now
	_, err = s.RBAC.UpdateUserRole(s.Ctx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:           billingManager.TwitchId,
		CompanyId:          company.Id,
		Role:               "Developer",
		RequestingTwitchId: owner.TwitchId,
	})
	s.NoError(err)

}

func (s *CompanySuite) Test_UpdateUserRole_ByWhitelistedAdmin() {
	company := s.createCompany()

	membership := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Developer",
	})

	_, err := s.RBAC.UpdateUserRole(s.AdminCtx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:  membership.TwitchId,
		Role:      "Marketer",
		CompanyId: company.Id,
	})
	s.NoError(err)
	s.checkAuthedEntityActionsLen(membership.TwitchId, "UserRole", WhitelistAdminTwitchId, company.Id, 2)

	// Check that the new member can be found with the new role
	membReload := s.membershipWithRole(company.Id, "Marketer")
	s.Equal(membership.TwitchId, membReload.TwitchId)
}

func (s *CompanySuite) Test_UpdateUserRole_FromShadowAccountToOther() {
	company := s.createCompany()

	resp, err := s.RBACAdmin.CreateShadowAccount(s.AdminCtx, &rbacrpc.CreateShadowAccountRequest{
		ClientId:  "client_id",
		CompanyId: company.Id,
		UserInput: []*rbacrpc.CreateShadowAccountUserInput{
			&rbacrpc.CreateShadowAccountUserInput{
				Email: "testemail@gmail.com",
			},
		},
	})
	s.NoError(err)
	s.NotNil(resp)
	s.NotEmpty(resp.Records)
	s.Equal(1, len(resp.Records))

	_, err = s.RBAC.UpdateUserRole(s.AdminCtx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:  resp.Records[0].TwitchId,
		Role:      "Marketer",
		CompanyId: company.Id,
	})
	s.Error(err)
}

func (s *CompanySuite) Test_UpdateUserRole_FromOtherToShadowAccount() {
	company := s.createCompany()

	membership := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Marketer",
	})

	_, err := s.RBAC.UpdateUserRole(s.AdminCtx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:  membership.TwitchId,
		Role:      "Shadow_Account",
		CompanyId: company.Id,
	})
	s.Error(err)
}

func (s *CompanySuite) Test_UpdateUser() {
	company := s.createCompany()

	memb := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Developer",
	})

	updateReq := &rbacrpc.UpdateUserRequest{
		TwitchId:  memb.TwitchId,
		CompanyId: company.Id,
		FirstName: "YOLOisMYname",
		LastName:  "LastYOLO",
		DevTitle:  "YOLOdeployer",
		DevEmail:  "yolo@yolo.yolo",
	}

	// End users (s.Ctx) cannot update user data (there's no RequestingTwitchId param atm)
	_, err := s.RBAC.UpdateUser(s.Ctx, updateReq)
	s.EqualErrorCode(err, twirp.PermissionDenied)

	// Whitelisted admins (s.AdminCtx) can update
	_, err = s.RBAC.UpdateUser(s.AdminCtx, updateReq)
	s.NoError(err)
	s.checkAuthedEntityActionsLen(memb.TwitchId, "UserRole", WhitelistAdminTwitchId, company.Id, 1)

	devReload := s.membershipWithRole(company.Id, "Developer")
	s.Equal(updateReq.FirstName, devReload.FirstName)
	s.Equal(updateReq.DevEmail, devReload.DevEmail)

	// Check that it fails with empty fields
	updateReq.FirstName = ""
	_, err = s.RBAC.UpdateUser(s.AdminCtx, updateReq)
	s.EqualErrorCode(err, twirp.InvalidArgument)
}

func (s *CompanySuite) Test_GetAllGamesWithCompanies() {
	company := s.createCompany()
	s.createOnboardedGame(company.Id)

	resp, err := s.RBAC.GetAllGamesWithCompanies(s.Ctx, &rbacrpc.GetAllGamesWithCompaniesRequest{})
	s.NoError(err)
	s.NotNil(resp.Games)        // don't check the exact length so the tests can be run on the same db
	s.NotZero(int(resp.XTotal)) // don't check the exact length so the tests can be run on the same db
}

func (s *CompanySuite) Test_CreateCompany_AlreadyOwnedGames_SameCompany() {
	company := s.createCompany()
	game1 := s.createOnboardedGame(company.Id)

	// make another application to the same game
	_, err := s.RBAC.CreateGameApplication(s.Ctx, &rbacrpc.CreateGameApplicationRequest{
		GameId:    game1.Id,
		CompanyId: company.Id,
	})
	s.EqualTwirp(err, twirp.AlreadyExists, "game already owned by a company")

	resp, err := s.RBAC.GetGameApplicationsByCompany(s.Ctx, &rbacrpc.Id{Id: company.Id})
	s.NoError(err)
	s.Equal(int(resp.XTotal), 0) // no new applications were added
}
func (s *CompanySuite) Test_CreateCompany_AlreadyOwnedGames_DifferentCompany() {
	companyA := s.createCompany()
	game1 := s.createOnboardedGame(companyA.Id)

	companyB := s.createCompany()
	_, err := s.RBAC.CreateGameApplication(s.Ctx, &rbacrpc.CreateGameApplicationRequest{
		GameId:    game1.Id,
		CompanyId: companyB.Id,
	})
	s.EqualTwirp(err, twirp.AlreadyExists, "game already owned by a company")

	resp, err := s.RBAC.GetGameApplicationsByCompany(s.Ctx, &rbacrpc.Id{Id: companyA.Id})
	s.NoError(err)
	s.Equal(int(resp.XTotal), 0) // no new applications were added
}
func (s *CompanySuite) Test_CreateCompanyResource_GetAll() {
	company := s.createCompany()
	resourcesToAdd := 3
	_, resourceTypes := s.createResources(company, resourcesToAdd, 2)

	// All resources
	resp, err := s.RBAC.ListResources(s.Ctx, &rbacrpc.ListResourcesRequest{
		CompanyId: company.Id,
	})

	totalAdded := len(resourceTypes) * resourcesToAdd

	s.NoError(err)
	s.Equal(int(resp.XTotal), totalAdded)
	s.Len(resp.Resources, totalAdded)
}

func (s *CompanySuite) Test_CreateCompanyResource_GetAll_Pagination() {
	company := s.createCompany()
	resourcesToAdd := 3
	_, resourceTypes := s.createResources(company, resourcesToAdd, 2)

	// All resources
	limit := uint64(1)
	offset := uint64(0)
	totalAdded := len(resourceTypes) * resourcesToAdd
	for ; offset < uint64(totalAdded); offset++ {
		resp, err := s.RBAC.ListResources(s.Ctx, &rbacrpc.ListResourcesRequest{
			CompanyId: company.Id,
			Offset:    offset,
			Limit:     limit,
		})

		s.NoError(err)
		s.Equal(int(resp.XTotal), totalAdded)
		s.Len(resp.Resources, int(limit))
	}

	// Make sure we went through the pages we expect
	s.Equal(int(offset), totalAdded)
}

func (s *CompanySuite) Test_CreateCompanyResource_GetByType() {
	company := s.createCompany()
	resourcesToAdd := 3
	_, resourceTypes := s.createResources(company, resourcesToAdd, 2)

	for _, resourceType := range resourceTypes {
		resp, err := s.RBAC.ListResources(s.Ctx, &rbacrpc.ListResourcesRequest{
			CompanyId:    company.Id,
			ResourceType: resourceType,
		})

		s.NoError(err)
		s.Equal(int(resp.XTotal), resourcesToAdd)
		s.Len(resp.Resources, resourcesToAdd)
	}
}

func (s *CompanySuite) Test_CreateCompanyResource_GetByType_Pagination() {
	company := s.createCompany()
	resourcesToAdd := 3
	_, resourceTypes := s.createResources(company, resourcesToAdd, 3)
	resourceType := resourceTypes[0]

	// All resources
	limit := uint64(1)
	offset := uint64(0)
	for ; offset < uint64(resourcesToAdd); offset++ {
		resp, err := s.RBAC.ListResources(s.Ctx, &rbacrpc.ListResourcesRequest{
			CompanyId:    company.Id,
			ResourceType: resourceType,
			Offset:       offset,
			Limit:        limit,
		})

		s.NoError(err)
		s.Equal(int(resp.XTotal), resourcesToAdd)
		s.Len(resp.Resources, int(limit))
	}

	// Make sure we went through the pages we expect
	s.Equal(int(offset), resourcesToAdd)
}
func (s *CompanySuite) Test_CreateCompanyResource_DuplicateResource() {
	company := s.createCompany()

	resource := &rbacrpc.Resource{
		ExternalId: randomString(16),
		Type:       "extension",
	}

	// Adding first time succeeds
	_, err := s.RBAC.CreateResource(s.Ctx, &rbacrpc.CreateResourceRequest{
		CompanyId: company.Id,
		Resource:  resource,
	})
	s.NoError(err)

	// Adding second time should fail
	_, err = s.RBAC.CreateResource(s.Ctx, &rbacrpc.CreateResourceRequest{
		CompanyId: company.Id,
		Resource:  resource,
	})
	s.EqualError(err, "twirp error already_exists: extension already owned by a company")
}

func (s *CompanySuite) Test_ListCompanies_OwnsGame_GameDoesntExist() {
	resp, err := s.RBAC.ListCompanies(s.Ctx, &rbacrpc.ListCompaniesRequest{
		OwnsGameId: unknownRandomGameID(),
	})
	s.NoError(err)
	s.Equal(int32(0), resp.XTotal)
}

func (s *CompanySuite) Test_ListCompanies_OwnsGame() {
	company := s.createCompany()
	game := s.createOnboardedGame(company.Id)

	resp, err := s.RBAC.ListCompanies(s.Ctx, &rbacrpc.ListCompaniesRequest{
		OwnsGameId: game.Id,
	})
	s.NoError(err)
	s.Equal(int32(1), resp.XTotal)
	s.Equal(company.Id, resp.Companies[0].Id)
}

func (s *CompanySuite) Test_ListCompanyMemberships() {
	company := s.createCompany()
	ownerMemb := s.membershipWithRole(company.Id, "Owner")

	type params struct {
		twitchID        string
		role            string
		canLeave        bool
		canAddMember    bool
		canRemoveMember bool
	}

	checkListCompaniesByTwitchID := func(ctx context.Context, p params) {
		resp, err := s.RBAC.ListCompanyMemberships(ctx, &rbacrpc.ListCompanyMembershipsRequest{TwitchId: p.twitchID})
		s.NoError(err)
		s.Len(resp.Memberships, 1)
		membership := resp.Memberships[0]

		s.Equal(company.Id, membership.Company.Id)
		s.Equal(p.role, membership.Role)
		s.Equal(p.canLeave, membership.SelfCanLeave, "can leave: "+p.role)
		s.Equal(p.canAddMember, membership.SelfCanAddMember, "can add member: "+p.role)
		s.Equal(p.canRemoveMember, membership.SelfCanRemoveMember, "can remove member: "+p.role)
		s.NotEmpty(membership.JoinedAt)
	}

	// The last admin (owner) can't leave, but can add other users
	checkListCompaniesByTwitchID(s.Ctx, params{
		role:            "Owner",
		twitchID:        ownerMemb.TwitchId,
		canLeave:        false,
		canAddMember:    true,
		canRemoveMember: true,
	})

	createAndCheck := func(p params) {
		memb := s.mustCreateMembership(membershipParams{
			CompanyID: company.Id,
			Role:      p.role,
		})
		p.twitchID = memb.TwitchId
		checkListCompaniesByTwitchID(s.Ctx, p)
	}

	createAndCheck(params{
		role:            "Administrator",
		canLeave:        true,
		canAddMember:    true,
		canRemoveMember: true,
	})
	createAndCheck(params{
		role:            "Billing_Manager",
		canLeave:        false,
		canAddMember:    true,
		canRemoveMember: true,
	})
	createAndCheck(params{
		role:            "Manager",
		canLeave:        true,
		canAddMember:    true,
		canRemoveMember: false,
	})
	createAndCheck(params{
		role:            "Marketer",
		canLeave:        true,
		canAddMember:    false,
		canRemoveMember: false,
	})
	createAndCheck(params{
		role:            "Developer",
		canLeave:        true,
		canAddMember:    false,
		canRemoveMember: false,
	})
}

func (s *CompanySuite) Test_UpdateCompany() {
	companyA := s.createCompany()
	companyB := s.createCompany()

	newURL := randomString(10)
	newName := "Updated newname COOL"
	_, err := s.RBAC.UpdateCompany(s.AdminCtx, &rbacrpc.UpdateCompanyRequest{
		Id:                companyA.Id,
		Identifier:        companyA.Identifier,
		CompanyName:       newName,
		Url:               newURL,
		Type:              companyA.Type,
		VhsContractSigned: companyA.VhsContractSigned,
		CampaignsEnabled:  companyA.CampaignsEnabled,
		CurseCompanyId:    companyA.CurseCompanyId,
	})
	s.NoError(err)

	reloadCompanyA, err := s.RBAC.GetCompany(s.Ctx, &rbacrpc.Id{Id: companyA.Id})
	s.NoError(err)
	reloadCompanyB, err := s.RBAC.GetCompany(s.Ctx, &rbacrpc.Id{Id: companyB.Id})
	s.NoError(err)

	// Check that it was updated as expected
	s.Equal(newURL, reloadCompanyA.Url)
	s.Equal(newName, reloadCompanyA.CompanyName)
	s.NotEqual(companyA.Url, reloadCompanyA.Url)
	s.NotEqual(companyA.CompanyName, reloadCompanyA.CompanyName)
	s.Equal(companyA.Identifier, reloadCompanyA.Identifier)
	s.Equal(companyA.Type, reloadCompanyA.Type)

	// And make sure other companies were not updated by mistake
	s.NotEqual(newURL, reloadCompanyB.Url)
	s.NotEqual(newName, reloadCompanyB.CompanyName)
	s.Equal(companyB.Url, reloadCompanyB.Url)
	s.Equal(companyB.CompanyName, reloadCompanyB.CompanyName)
	s.Equal(companyB.Identifier, reloadCompanyB.Identifier)
	s.Equal(companyB.Type, reloadCompanyB.Type)
}

func (s *CompanySuite) Test_UpdateCompany_InvalidCompanyID() {
	_, err := s.RBAC.UpdateCompany(s.AdminCtx, &rbacrpc.UpdateCompanyRequest{
		Id:          "not-a-valid-uuid",
		Identifier:  "foobar",
		CompanyName: "FooBar",
	})
	s.EqualErrorCode(err, twirp.InvalidArgument)
	s.EqualErrorMsg(err, "id invalid UUID format")

	_, err = s.RBAC.UpdateCompany(s.AdminCtx, &rbacrpc.UpdateCompanyRequest{
		Id:          common.NewUUID(), // valid uuid
		Identifier:  "foobar",
		CompanyName: "FooBar",
	})
	s.EqualErrorCode(err, twirp.InvalidArgument)
	s.EqualErrorMsg(err, "id does not identify an existing company")
}

func (s *CompanySuite) Test_UpdateCompany_NonWhitelistedAdminPermissionDenied() {
	company := s.createCompany()
	_, err := s.RBAC.UpdateCompany(s.Ctx, &rbacrpc.UpdateCompanyRequest{
		Id:          company.Id,
		Identifier:  "foobar",
		CompanyName: "FooBar",
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)
}

func (s *CompanySuite) Test_SearchCompanies() {
	company := s.createCompany()

	result, err := s.RBAC.SearchCompanies(s.Ctx, &rbacrpc.SearchCompaniesRequest{
		Query: company.CompanyName,
	})
	s.NoError(err)

	if result.XTotal < 1 {
		s.Fail("expected XTotal to be at least 1, got %d", result.XTotal)
	}

	foundCompany := false
	for _, searchedCompany := range result.Companies {
		if searchedCompany.Id == company.Id {
			foundCompany = true
		}
	}

	s.True(foundCompany)
}

func (s *CompanySuite) Test_DeleteCompany() {
	company := s.createCompany()
	companyOwner := s.membershipWithRole(company.Id, "Owner")
	OtherCompany := s.createCompany()

	// Remove last user in company (is possible as whitelisted super-admin)
	_, err := s.RBAC.RemoveCompanyMembership(s.AdminCtx, &rbacrpc.RemoveCompanyMembershipRequest{
		TwitchId:  companyOwner.TwitchId,
		CompanyId: company.Id,
	})
	s.NoError(err)

	// Should have 0 users in company
	usersResp, err := s.RBAC.GetUsersByCompanyId(s.AdminCtx, &rbacrpc.GetUsersByCompanyIdRequest{Id: company.Id})
	s.NoError(err)
	s.Len(usersResp.Memberships, 0)

	// There should still be one user in company 2
	// Verifies we didn't delete everyone in other companies by mistake
	usersResp, err = s.RBAC.GetUsersByCompanyId(s.AdminCtx, &rbacrpc.GetUsersByCompanyIdRequest{Id: OtherCompany.Id})
	s.NoError(err)
	s.Len(usersResp.Memberships, 1)

	// Delete company
	_, err = s.RBACAdmin.DeleteCompany(s.AdminCtx, &rbacrpc.DeleteCompanyRequest{Id: company.Id})
	s.NoError(err)
	_, err = s.RBAC.GetCompany(s.Ctx, &rbacrpc.Id{Id: company.Id})
	s.EqualErrorCode(err, twirp.NotFound)

	// OtherCompany still exists
	// Verifies we didn't delete all companies by mistake
	otherCompany2, err := s.RBAC.GetCompany(s.Ctx, &rbacrpc.Id{Id: OtherCompany.Id})
	s.NoError(err)
	s.Equal(OtherCompany.Id, otherCompany2.Id)
}
