package e2e

import (
	"testing"

	"code.justin.tv/devrel/devsite-rbac/models"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/stretchr/testify/suite"
	"github.com/twitchtv/twirp"
)

func Test_CompanyApplicationsSuite(t *testing.T) {
	s := &CompanyApplicationsSuite{}
	suite.Run(t, s)
}

type CompanyApplicationsSuite struct {
	Suite
}

func (s *CompanyApplicationsSuite) Test_ListCompanyApplications() {
	companyAppID := s.createCompanyApplication()

	companyApp, err := s.RBAC.GetCompanyApplication(s.Ctx, &rbacrpc.Id{Id: companyAppID})
	s.NoError(err)
	s.NotNil(companyApp)

	listResp1, err := s.RBAC.ListCompanyApplications(s.AdminCtx, &rbacrpc.ListCompanyApplicationsRequest{
		TwitchId: companyApp.TwitchId,
	})
	s.NoError(err)
	s.Len(listResp1.CompanyApplications, 1)
}

func (s *CompanyApplicationsSuite) Test_CreateCompanyApplication_WithGames() {
	req := newCompanyApplicationRequest()
	req.Games = []string{"1", "2", "3"}
	companyApp, err := s.RBAC.CreateCompanyApplication(s.Ctx, req)
	s.NoError(err)
	companyAppID := companyApp.Id // ID is the only field returned

	// Load and check that the games are present
	reCompanyApp, err := s.RBAC.GetCompanyApplication(s.AdminCtx, &rbacrpc.Id{Id: companyAppID})
	s.NoError(err)
	s.Len(reCompanyApp.Games, 3)
	s.Equal("1", reCompanyApp.Games[0].Id)
	s.Equal("1", reCompanyApp.Games[0].Name) // the Discovery NOOP fakes the name from the ID
}

func (s *CompanyApplicationsSuite) Test_DeleteCompanyApplication() {
	companyAppID := s.createCompanyApplication()

	_, err := s.RBAC.DeleteCompanyApplication(s.AdminCtx, &rbacrpc.DeleteCompanyApplicationRequest{
		Id: companyAppID,
	})
	s.NoError(err)

	resp, err := s.RBAC.GetCompanyApplication(s.AdminCtx, &rbacrpc.Id{
		Id: companyAppID,
	})
	s.NoError(err)
	s.NotNil(resp)

	s.Equal(int32(models.CompanyApplicationRejected), resp.Status)
}

func (s *CompanyApplicationsSuite) Test_CreateCompanyApplication_WithExistingUser() {
	// Ensure that existing twitch developers are not duplicated during onboarding
	previousCompany := s.createCompany()
	origUser := s.membershipWithRole(previousCompany.Id, "Owner")

	companyAppRequest := newCompanyApplicationRequest()
	companyAppRequest.TwitchId = origUser.TwitchId

	companyApp, err := s.RBAC.CreateCompanyApplication(s.Ctx, companyAppRequest)
	s.NoError(err)

	company, err := s.RBAC.OnboardCompany(s.AdminCtx, &rbacrpc.OnboardCompanyRequest{
		Id: companyApp.Id,
	})
	s.NoError(err)

	counts := s.membershipRoleCounts(company.Id)
	s.Equal(counts.Owners, 1)
	admin := s.membershipWithRole(company.Id, "Owner")
	s.Equal(origUser.TwitchId, admin.TwitchId)
}

func (s *CompanyApplicationsSuite) Test_UpdateCompanyApplication() {
	companyAppId := s.createCompanyApplication()

	originalCompanyApp, err := s.RBAC.GetCompanyApplication(s.Ctx, &rbacrpc.Id{Id: companyAppId})
	s.NoError(err)

	_, err = s.RBAC.UpdateCompanyApplication(s.AdminCtx, &rbacrpc.UpdateCompanyApplicationRequest{
		Id:               companyAppId,
		CompanyName:      "New Updated Name",
		ContactFirstName: "New Updated First Name",
		ContactLastName:  "New Updated Last Name",

		CompanyWebsite:  originalCompanyApp.CompanyWebsite,
		CompanyType:     originalCompanyApp.CompanyType,
		Industry:        originalCompanyApp.Industry,
		CompanySize:     originalCompanyApp.CompanySize,
		City:            originalCompanyApp.City,
		State:           originalCompanyApp.State,
		Country:         originalCompanyApp.Country,
		ContactTitle:    originalCompanyApp.ContactTitle,
		ContactEmail:    originalCompanyApp.ContactEmail,
		ProductInterest: originalCompanyApp.ProductInterest,
		JoinReason:      originalCompanyApp.JoinReason,
		TwitchId:        originalCompanyApp.TwitchId,
		Status:          originalCompanyApp.Status,
	})
	s.NoError(err)

	updatedCompanyApp, err := s.RBAC.GetCompanyApplication(s.Ctx, &rbacrpc.Id{Id: companyAppId})
	s.NoError(err)

	s.Equal("New Updated Name", updatedCompanyApp.CompanyName)
	s.Equal("New Updated First Name", updatedCompanyApp.ContactFirstName)
	s.Equal("New Updated Last Name", updatedCompanyApp.ContactLastName)
	s.NotEqual(originalCompanyApp.CompanyName, updatedCompanyApp.CompanyName)

	s.Equal(originalCompanyApp.Industry, updatedCompanyApp.Industry)
	s.Equal(originalCompanyApp.CompanySize, updatedCompanyApp.CompanySize)
	s.Equal(originalCompanyApp.City, updatedCompanyApp.City)
	s.Equal(originalCompanyApp.ContactEmail, updatedCompanyApp.ContactEmail)
	s.Equal(originalCompanyApp.TwitchId, updatedCompanyApp.TwitchId)

	s.Equal(originalCompanyApp.CreatedAt, updatedCompanyApp.CreatedAt)
	s.NotEqual(originalCompanyApp.UpdatedAt, updatedCompanyApp.UpdatedAt)
}

func (s *CompanyApplicationsSuite) Test_OnboardCompany_ApplicationNotFound() {
	_, err := s.RBAC.OnboardCompany(s.AdminCtx, &rbacrpc.OnboardCompanyRequest{
		Id: "00000000-0000-0000-0000-000000000123", // non existing
	})
	s.NotNil(err)
	s.EqualTwirp(err, twirp.InvalidArgument, "id company application not found")
}

func (s *CompanyApplicationsSuite) Test_OnboardCompany_OwnerIsWhitelisted() {
	companyAppRequest := newCompanyApplicationRequest()

	companyApp, err := s.RBAC.CreateCompanyApplication(s.Ctx, companyAppRequest)
	s.NoError(err)

	updateFeatureGatingValueRequest := &rbacrpc.UpdateFeatureGatingValueRequest{
		Key:           models.NewOrgConsoleUserWhitelistKey,
		ValueInString: companyAppRequest.TwitchId,
	}

	_, err = s.RBACAdmin.UpdateFeatureGatingValue(s.AdminCtx, updateFeatureGatingValueRequest)
	s.NoError(err)

	_, err = s.RBAC.OnboardCompany(s.AdminCtx, &rbacrpc.OnboardCompanyRequest{
		Id: companyApp.Id,
	})
	s.NoError(err)

	resp, err := s.RBAC.IsValueEnabled(s.Ctx, &rbacrpc.IsValueEnabledRequest{
		FeatureGatingKey: models.NewOrgConsoleOrgWhitelistKey,
		ValueInString:    companyApp.Id,
	})
	s.NoError(err)
	s.Equal(true, resp.Enabled)
}

func (s *CompanyApplicationsSuite) Test_OnboardCompany_OwnerIsNotWhitelisted() {
	companyAppRequest := newCompanyApplicationRequest()

	companyApp, err := s.RBAC.CreateCompanyApplication(s.Ctx, companyAppRequest)
	s.NoError(err)

	_, err = s.RBAC.OnboardCompany(s.AdminCtx, &rbacrpc.OnboardCompanyRequest{
		Id: companyApp.Id,
	})
	s.NoError(err)

	resp, err := s.RBAC.IsValueEnabled(s.Ctx, &rbacrpc.IsValueEnabledRequest{
		FeatureGatingKey: models.NewOrgConsoleOrgWhitelistKey,
		ValueInString:    companyApp.Id,
	})
	s.NoError(err)
	s.Equal(false, resp.Enabled)
}

func (s *CompanyApplicationsSuite) Test_SearchCompanyApplications() {
	companyApp := s.createAndGetCompanyApplication()

	resp, err := s.RBAC.SearchCompanyApplications(s.Ctx, &rbacrpc.SearchCompanyApplicationsRequest{
		Query: companyApp.CompanyName,
		Limit: 1,
	})
	s.NoError(err)
	s.Len(resp.CompanyApplications, 1)
	s.Equal(companyApp.CompanyName, resp.CompanyApplications[0].CompanyName)
	s.Equal(resp.XTotal, int32(1))
}
