package e2e

import (
	"testing"

	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/stretchr/testify/suite"
)

type FeatureGatingTestSuite struct {
	Suite
}

const (
	TEST_BOOL_FEATURE_GATING_KEY    = "feature_gating_bool_test_enabled"
	TEST_STRINGS_FEATURE_GATING_KEY = "feature_gating_test_strings"
)

func Test_FeatureGatingTestSuite(t *testing.T) {
	s := &FeatureGatingTestSuite{}
	suite.Run(t, s)
}

func (s *FeatureGatingTestSuite) Test_GetFeatureGatingValueInBool_caller_not_admin() {
	s.AdminCtx = s.withAuth(s.AdminCtx, "test_twitch_id")
	mockRequest := &rbacrpc.GetFeatureGatingValueRequest{
		Key: TEST_BOOL_FEATURE_GATING_KEY,
	}

	_, err := s.RBACAdmin.GetFeatureGatingValueInBool(s.AdminCtx, mockRequest)
	s.Error(err)
}

func (s *FeatureGatingTestSuite) Test_GetFeatureGatingValueInBool() {
	mockRequest := &rbacrpc.GetFeatureGatingValueRequest{
		Key: TEST_BOOL_FEATURE_GATING_KEY,
	}

	resp, err := s.RBACAdmin.GetFeatureGatingValueInBool(s.AdminCtx, mockRequest)
	s.NoError(err)
	s.Equal(true, resp.ValueInBool)
}

func (s *FeatureGatingTestSuite) Test_GetFeatureGatingValueInBool_Key_not_exist() {
	mockRequest := &rbacrpc.GetFeatureGatingValueRequest{
		Key: "any bool key",
	}

	resp, err := s.RBACAdmin.GetFeatureGatingValueInBool(s.AdminCtx, mockRequest)
	s.NoError(err)
	s.Equal(false, resp.ValueInBool)
}

func (s *FeatureGatingTestSuite) Test_SetFeatureGatingBoolValueInBool_insert_new_key() {
	testKey := "feature_gating_test_set_value"

	mockRequest := &rbacrpc.SetFeatureGatingBoolValueRequest{
		Key:         testKey,
		ValueInBool: true,
	}

	_, err := s.RBACAdmin.SetFeatureGatingBoolValue(s.AdminCtx, mockRequest)
	s.NoError(err)

	mockGetRequest := &rbacrpc.GetFeatureGatingValueRequest{
		Key: testKey,
	}

	resp, err := s.RBACAdmin.GetFeatureGatingValueInBool(s.AdminCtx, mockGetRequest)
	s.NoError(err)
	s.Equal(true, resp.ValueInBool)
}

func (s *FeatureGatingTestSuite) Test_SetFeatureGatingBoolValueInBool_update_existing_key() {
	mockRequest := &rbacrpc.SetFeatureGatingBoolValueRequest{
		Key:         TEST_BOOL_FEATURE_GATING_KEY,
		ValueInBool: false,
	}

	_, err := s.RBACAdmin.SetFeatureGatingBoolValue(s.AdminCtx, mockRequest)
	s.NoError(err)

	mockGetRequest := &rbacrpc.GetFeatureGatingValueRequest{
		Key: TEST_BOOL_FEATURE_GATING_KEY,
	}

	resp, err := s.RBACAdmin.GetFeatureGatingValueInBool(s.AdminCtx, mockGetRequest)
	s.NoError(err)
	s.Equal(false, resp.ValueInBool)
}

func (s *FeatureGatingTestSuite) Test_GetFeatureGatingValueInStrings_caller_not_admin() {
	mockRequest := &rbacrpc.GetFeatureGatingValueRequest{
		Key: TEST_STRINGS_FEATURE_GATING_KEY,
	}

	_, err := s.RBACAdmin.GetFeatureGatingValueInStrings(s.Ctx, mockRequest)
	s.Error(err)
}

func (s *FeatureGatingTestSuite) Test_GetFeatureGatingValueInStrings() {
	mockRequest := &rbacrpc.GetFeatureGatingValueRequest{
		Key: TEST_STRINGS_FEATURE_GATING_KEY,
	}

	resp, err := s.RBACAdmin.GetFeatureGatingValueInStrings(s.AdminCtx, mockRequest)
	s.NoError(err)
	s.Equal(3, len(resp.ValuesInString))
	s.Equal("fg_test_1", resp.ValuesInString[0])
	s.Equal("fg_test_2", resp.ValuesInString[1])
	s.Equal("fg_test_3", resp.ValuesInString[2])
}
func (s *FeatureGatingTestSuite) Test_GetFeatureGatingValueInStrings_Key_not_exist() {
	mockRequest := &rbacrpc.GetFeatureGatingValueRequest{
		Key: "any_strings_key",
	}

	resp, err := s.RBACAdmin.GetFeatureGatingValueInStrings(s.AdminCtx, mockRequest)
	s.NoError(err)
	s.Equal(0, len(resp.ValuesInString))
}

func (s *FeatureGatingTestSuite) Test_GetFeatureGatingValueInStrings_insert_new_key() {
	testKey := "feature_gating_test_strings_set_value_1"

	mockRequest := &rbacrpc.UpdateFeatureGatingValueRequest{
		Key:           testKey,
		ValueInString: "test_new_fg_1",
		Action:        rbacrpc.UpdateFeatureGatingValueAction_ADD,
	}

	_, err := s.RBACAdmin.UpdateFeatureGatingValue(s.AdminCtx, mockRequest)
	s.NoError(err)

	mockGetRequest := &rbacrpc.GetFeatureGatingValueRequest{
		Key: testKey,
	}

	resp, err := s.RBACAdmin.GetFeatureGatingValueInStrings(s.AdminCtx, mockGetRequest)
	s.NoError(err)
	s.Equal(1, len(resp.ValuesInString))
	s.Equal("test_new_fg_1", resp.ValuesInString[0])
}

func (s *FeatureGatingTestSuite) Test_GetFeatureGatingValueInStrings_update_existing_key() {
	testKey := "feature_gating_test_strings_set_value_2"

	mockRequest := &rbacrpc.UpdateFeatureGatingValueRequest{
		Key:           testKey,
		ValueInString: "test_new_fg_1",
		Action:        rbacrpc.UpdateFeatureGatingValueAction_ADD,
	}

	resp, err := s.RBACAdmin.UpdateFeatureGatingValue(s.AdminCtx, mockRequest)
	s.NoError(err)
	s.Equal(1, len(resp.ValuesInString))
	s.Equal("test_new_fg_1", resp.ValuesInString[0])

	mockRequest = &rbacrpc.UpdateFeatureGatingValueRequest{
		Key:           testKey,
		ValueInString: "test_new_fg_2",
		Action:        rbacrpc.UpdateFeatureGatingValueAction_ADD,
	}
	resp, err = s.RBACAdmin.UpdateFeatureGatingValue(s.AdminCtx, mockRequest)
	s.NoError(err)
	s.Equal(2, len(resp.ValuesInString))

	mockRequest = &rbacrpc.UpdateFeatureGatingValueRequest{
		Key:           testKey,
		ValueInString: "test_new_fg_1",
		Action:        rbacrpc.UpdateFeatureGatingValueAction_REMOVE,
	}
	resp, err = s.RBACAdmin.UpdateFeatureGatingValue(s.AdminCtx, mockRequest)
	s.NoError(err)
	s.Equal(1, len(resp.ValuesInString))

	mockRequest = &rbacrpc.UpdateFeatureGatingValueRequest{
		Key:           testKey,
		ValueInString: "test_new_fg_2",
		Action:        rbacrpc.UpdateFeatureGatingValueAction_REMOVE,
	}
	resp, err = s.RBACAdmin.UpdateFeatureGatingValue(s.AdminCtx, mockRequest)
	s.NoError(err)
	s.Equal(0, len(resp.ValuesInString))
}

func (s *FeatureGatingTestSuite) Test_GetFeatureGatingValueInStrings_add_repeated_value() {
	testKey := "feature_gating_test_strings_set_value_3"

	mockRequest := &rbacrpc.UpdateFeatureGatingValueRequest{
		Key:           testKey,
		ValueInString: "test_new_fg_1",
		Action:        rbacrpc.UpdateFeatureGatingValueAction_ADD,
	}

	resp, err := s.RBACAdmin.UpdateFeatureGatingValue(s.AdminCtx, mockRequest)
	s.NoError(err)
	s.Equal(1, len(resp.ValuesInString))

	mockRequest = &rbacrpc.UpdateFeatureGatingValueRequest{
		Key:           testKey,
		ValueInString: "test_new_fg_1",
		Action:        rbacrpc.UpdateFeatureGatingValueAction_ADD,
	}

	_, err = s.RBACAdmin.UpdateFeatureGatingValue(s.AdminCtx, mockRequest)
	s.Error(err)
}

func (s *FeatureGatingTestSuite) Test_GetFeatureGatingValueInStrings_remove_nonexisting_value() {
	mockRequest := &rbacrpc.UpdateFeatureGatingValueRequest{
		Key:           TEST_STRINGS_FEATURE_GATING_KEY,
		ValueInString: "test_new_fg_100",
		Action:        rbacrpc.UpdateFeatureGatingValueAction_REMOVE,
	}

	_, err := s.RBACAdmin.UpdateFeatureGatingValue(s.AdminCtx, mockRequest)
	s.Error(err)
}

func (s *FeatureGatingTestSuite) Test_IsFeatureEnabled() {
	mockRequest := &rbacrpc.IsFeatureEnabledRequest{
		FeatureGatingKey: TEST_BOOL_FEATURE_GATING_KEY,
	}

	resp, err := s.RBAC.IsFeatureEnabled(s.Ctx, mockRequest)
	s.NoError(err)
	s.Equal(true, resp.Enabled)
}

func (s *FeatureGatingTestSuite) Test_IsFeatureEnabled_nonexisting_key() {
	mockRequest := &rbacrpc.IsFeatureEnabledRequest{
		FeatureGatingKey: "any key",
	}

	resp, err := s.RBAC.IsFeatureEnabled(s.Ctx, mockRequest)
	s.NoError(err)
	s.Equal(false, resp.Enabled)
}

func (s *FeatureGatingTestSuite) Test_IsValueEnabled() {
	mockRequest := &rbacrpc.IsValueEnabledRequest{
		FeatureGatingKey: TEST_STRINGS_FEATURE_GATING_KEY,
		ValueInString:    "fg_test_1",
	}

	resp, err := s.RBAC.IsValueEnabled(s.Ctx, mockRequest)
	s.NoError(err)
	s.Equal(true, resp.Enabled)
}

func (s *FeatureGatingTestSuite) Test_IsValueEnabled_nonexsiting_key() {
	mockRequest := &rbacrpc.IsValueEnabledRequest{
		FeatureGatingKey: "any key any key",
		ValueInString:    "fg_test_1",
	}

	resp, err := s.RBAC.IsValueEnabled(s.Ctx, mockRequest)
	s.NoError(err)
	s.Equal(false, resp.Enabled)
}
