package e2e

import (
	"testing"

	"code.justin.tv/devrel/devsite-rbac/clients/users"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"

	"github.com/stretchr/testify/suite"
	"github.com/twitchtv/twirp"
)

type GetExtensionBillingManagerTestSuite struct {
	Suite
}

func Test_GetExtensionBillingManagerTestSuite(t *testing.T) {
	s := &GetExtensionBillingManagerTestSuite{}
	suite.Run(t, s)
}

type testData struct {
	testCompanyID        string
	testBillingManagerID string
	testExtensionID      string
}

func (s *GetExtensionBillingManagerTestSuite) mockTestData(isBillingManagerDeleted bool) *testData {
	company := s.createCompany()
	testExtensionId := randomString(16)

	resource := &rbacrpc.Resource{
		ExternalId: testExtensionId,
		Type:       "extension",
	}

	_, err := s.RBAC.CreateResource(s.AdminCtx, &rbacrpc.CreateResourceRequest{
		CompanyId: company.Id,
		Resource:  resource,
	})
	s.NoError(err)

	var testBillingManagerId = randomTwitchIDWithTIMsEnabled()
	if isBillingManagerDeleted {
		testBillingManagerId = users.MockedDeletedUserTwitchID
	}
	s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Billing_Manager",
		TwitchID:  testBillingManagerId,
	})

	_, err = s.RBAC.SetExtensionBillingManager(s.AdminCtx, &rbacrpc.SetExtensionBillingManagerRequest{
		BillingManagerTwitchId: testBillingManagerId,
		ExtensionClientId:      testExtensionId,
	})
	s.NoError(err)

	return &testData{
		testCompanyID:        company.Id,
		testBillingManagerID: testBillingManagerId,
		testExtensionID:      testExtensionId,
	}
}

func (s *GetExtensionBillingManagerTestSuite) SetupSuite() {
	s.Suite.SetupSuite()
}

func (s *GetExtensionBillingManagerTestSuite) Test_GetExtensionBillingManager() {
	td := s.mockTestData(false)
	// Add developer
	testRandomUserId := randomTwitchID()
	s.mustCreateMembership(membershipParams{
		CompanyID: td.testCompanyID,
		Role:      "Developer",
		TwitchID:  testRandomUserId,
	})
	// whitelisted user can access
	bm, err := s.RBAC.GetExtensionBillingManager(s.AdminCtx, &rbacrpc.GetExtensionBillingManagerRequest{
		ExtensionClientId: td.testExtensionID,
	})
	s.NoError(err)

	s.EqualValues(
		&rbacrpc.ExtensionBillingManager{
			ExtensionClientId:      td.testExtensionID,
			BillingManagerTwitchId: td.testBillingManagerID,
		},
		&rbacrpc.ExtensionBillingManager{
			ExtensionClientId:      bm.ExtensionBillingManager.ExtensionClientId,
			BillingManagerTwitchId: bm.ExtensionBillingManager.BillingManagerTwitchId,
		},
	)
	s.Equal(false, bm.IsBillingManagerAccountDeleted)

	// developer can access
	_, err = s.RBAC.GetExtensionBillingManager(s.Ctx, &rbacrpc.GetExtensionBillingManagerRequest{
		ExtensionClientId:  td.testExtensionID,
		RequestingTwitchId: testRandomUserId,
	})
	s.NoError(err)

	// random extension id which is not part of company returns found
	_, err = s.RBAC.GetExtensionBillingManager(s.Ctx, &rbacrpc.GetExtensionBillingManagerRequest{
		ExtensionClientId:  randomString(16),
		RequestingTwitchId: testRandomUserId,
	})
	s.EqualErrorCode(err, twirp.NotFound)

	// random user whos not apart of the company can't access
	_, err = s.RBAC.GetExtensionBillingManager(s.Ctx, &rbacrpc.GetExtensionBillingManagerRequest{
		ExtensionClientId:  td.testExtensionID,
		RequestingTwitchId: randomTwitchID(),
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)

	// no rows were found
	_, err = s.RBAC.GetExtensionBillingManager(s.AdminCtx, &rbacrpc.GetExtensionBillingManagerRequest{
		ExtensionClientId: randomString(16),
	})
	s.EqualErrorCode(err, twirp.NotFound)
}

func (s *GetExtensionBillingManagerTestSuite) Test_GetExtensionBillingManager_DeletedTwitchAccount() {
	td := s.mockTestData(true)
	// whitelisted user can access
	bm, err := s.RBAC.GetExtensionBillingManager(s.AdminCtx, &rbacrpc.GetExtensionBillingManagerRequest{
		ExtensionClientId: td.testExtensionID,
	})
	s.NoError(err)
	s.EqualValues(
		&rbacrpc.ExtensionBillingManager{
			ExtensionClientId:      td.testExtensionID,
			BillingManagerTwitchId: td.testBillingManagerID,
		},
		&rbacrpc.ExtensionBillingManager{
			ExtensionClientId:      bm.ExtensionBillingManager.ExtensionClientId,
			BillingManagerTwitchId: bm.ExtensionBillingManager.BillingManagerTwitchId,
		},
	)
	s.Equal(true, bm.IsBillingManagerAccountDeleted)
}

func (s *GetExtensionBillingManagerTestSuite) Test_GetExtensionBillingManager_Params() {
	// non whitelisted user needs to specify requested twitch id
	_, err := s.RBAC.GetExtensionBillingManager(s.Ctx, &rbacrpc.GetExtensionBillingManagerRequest{
		ExtensionClientId: "any-extension-id",
	})
	s.EqualErrorCode(err, twirp.InvalidArgument)

	// users need to specify Extension Client Id
	_, err = s.RBAC.GetExtensionBillingManager(s.AdminCtx, &rbacrpc.GetExtensionBillingManagerRequest{
		RequestingTwitchId: randomTwitchID(),
	})
	s.EqualErrorCode(err, twirp.InvalidArgument)
}
