package e2e

import (
	"strings"
	"testing"

	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/stretchr/testify/suite"
	"github.com/twitchtv/twirp"
)

func Test_PrivacySuite(t *testing.T) {
	s := &PrivacySuite{}
	suite.Run(t, s)
}

type PrivacySuite struct {
	Suite
}

// TODO Add test to mock PDMS and verify it loops when PDMS is ready
func (s *PrivacySuite) Test_PrivacyIndividualDelete_OwnerTransition() {
	company := s.createCompany()
	role := "Administrator"
	owner := s.membershipWithRole(company.Id, "Owner")
	admin := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      role,
	})

	gameApp := s.createGameApplication(company.Id)
	companyResource := s.createOnboardedGame(company.Id)
	randomyCompany := s.createCompany()
	companyInvite :=
		s.mustCreateCompanyInvite(companyInviteParams{
			CompanyID:       randomyCompany.Id,
			InviteeTwitchID: owner.TwitchId,
			Role:            strings.ToLower(role),
		})

	//tmp to "use" these vars, delete when checking gameApp/companyResources below
	s.NotNil(gameApp)
	s.NotNil(companyResource)
	s.NotNil(companyInvite)

	//Add application history

	_, saErr := s.Privacy.StartAdmin(s.AdminCtx, &rbacrpc.Id{Id: owner.TwitchId})
	s.NoError(saErr)

	// Need to assert owner does not exist
	_, err := s.RBAC.GetMembership(s.AdminCtx, &rbacrpc.GetMembershipRequest{
		CompanyId: company.Id,
		TwitchId:  owner.TwitchId,
	})
	s.Error(err) //should have membership not found error

	// assert admin is Owner
	adminsNewMemb, err := s.RBAC.GetMembership(s.AdminCtx, &rbacrpc.GetMembershipRequest{
		CompanyId: company.Id,
		TwitchId:  admin.TwitchId,
	})
	s.NoError(err)
	s.Equal("Owner", adminsNewMemb.Role)
	s.checkAuthedEntityActionsLen(adminsNewMemb.TwitchId, "UserRole", WhitelistAdminTwitchId, company.Id, 2)

	// Verify no other users exist on the new company
	usersResp, err := s.RBAC.GetUsersByCompanyId(s.Ctx, &rbacrpc.GetUsersByCompanyIdRequest{Id: company.Id})
	s.NoError(err)

	counts := countMemberships(usersResp.Memberships)
	s.Equal(counts.Owners, 1)
	s.Equal(counts.Admins, 0)
	s.Equal(counts.Managers, 0)
	s.Equal(counts.Marketers, 0)
	s.Equal(counts.Developers, 0)

	// also check these are all empty
	afterGameResources, err := s.RBAC.GetGamesByTwitchID(s.Ctx, &rbacrpc.Id{Id: owner.TwitchId})
	s.Equal(int32(0), afterGameResources.XTotal)
	s.NoError(err)

	afterCompanyInvites, err := s.RBAC.GetCompanyInvitesByTwitchID(s.Ctx, &rbacrpc.GetCompanyInvitesByTwitchIDRequest{TwitchId: owner.TwitchId})
	s.Equal(int32(0), afterCompanyInvites.XTotal)
	s.NoError(err)

	// game application should still be intact because it's on the org, not the user
	afterGameApp, err := s.RBAC.GetGameApplication(s.Ctx, &rbacrpc.Id{Id: gameApp.Id})
	s.NoError(err)
	s.Equal(gameApp, afterGameApp)
}

func (s *PrivacySuite) Test_PrivacySoloOwnerDelete_CompanyDeletes() {
	company := s.createCompany()
	role := "Administrator"
	owner := s.membershipWithRole(company.Id, "Owner")

	gameApp := s.createGameApplication(company.Id)
	companyResource := s.createOnboardedGame(company.Id)
	randomyCompany := s.createCompany()
	companyInvite :=
		s.mustCreateCompanyInvite(companyInviteParams{
			CompanyID:       randomyCompany.Id,
			InviteeTwitchID: owner.TwitchId,
			Role:            strings.ToLower(role),
		})

	//tmp to "use" these vars, delete when checking gameApp/companyResources below
	s.NotNil(gameApp)
	s.NotNil(companyResource)
	s.NotNil(companyInvite)

	//Add application history

	_, saErr := s.Privacy.StartAdmin(s.AdminCtx, &rbacrpc.Id{Id: owner.TwitchId})
	s.NoError(saErr)

	// Need to assert owner does not exist
	_, err := s.RBAC.GetMembership(s.AdminCtx, &rbacrpc.GetMembershipRequest{
		CompanyId: company.Id,
		TwitchId:  owner.TwitchId,
	})
	s.Error(err) //should have membership not found error

	// Verify no other users exist on the new company
	usersResp, err := s.RBAC.GetUsersByCompanyId(s.Ctx, &rbacrpc.GetUsersByCompanyIdRequest{Id: company.Id})
	s.EqualErrorCode(err, twirp.NotFound)

	counts := countMemberships(usersResp.Memberships)
	s.Equal(counts.Owners, 0)
	s.Equal(counts.Admins, 0)
	s.Equal(counts.Managers, 0)
	s.Equal(counts.Marketers, 0)
	s.Equal(counts.Developers, 0)

	// also check these are all empty
	afterGameResources, err := s.RBAC.GetGamesByTwitchID(s.Ctx, &rbacrpc.Id{Id: owner.TwitchId})
	s.Equal(int32(0), afterGameResources.XTotal)
	s.NoError(err)

	afterCompanyInvites, err := s.RBAC.GetCompanyInvitesByTwitchID(s.Ctx, &rbacrpc.GetCompanyInvitesByTwitchIDRequest{TwitchId: owner.TwitchId})
	s.Equal(int32(0), afterCompanyInvites.XTotal)
	s.NoError(err)

	afterGameApp, err := s.RBAC.GetGameApplication(s.Ctx, &rbacrpc.Id{Id: gameApp.Id})
	s.EqualErrorCode(err, twirp.NotFound)
	s.Equal("", afterGameApp.Id)

	afterCompany, err := s.RBAC.GetCompany(s.Ctx, &rbacrpc.Id{Id: company.Id})
	s.EqualErrorCode(err, twirp.NotFound)
	s.Equal("", afterCompany.Id)
}

func (s *PrivacySuite) Test_PrivacyDeveloperDelete_DeleteJustMembership() {
	company := s.createCompany()
	owner := s.membershipWithRole(company.Id, "Owner")
	dev := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Developer",
	})

	// Verify 2 users in company (owner/dev)
	usersRespBefore, err := s.RBAC.GetUsersByCompanyId(s.Ctx, &rbacrpc.GetUsersByCompanyIdRequest{Id: company.Id})
	s.NoError(err)
	s.Equal(uint64(2), usersRespBefore.Total)

	counts := countMemberships(usersRespBefore.Memberships)
	s.Equal(counts.Owners, 1)
	s.Equal(counts.Admins, 0)
	s.Equal(counts.Managers, 0)
	s.Equal(counts.Marketers, 0)
	s.Equal(counts.Developers, 1)

	_, saErr := s.Privacy.StartAdmin(s.AdminCtx, &rbacrpc.Id{Id: dev.TwitchId})
	s.NoError(saErr)

	// Need to assert owner does not exist
	_, err = s.RBAC.GetMembership(s.AdminCtx, &rbacrpc.GetMembershipRequest{
		CompanyId: company.Id,
		TwitchId:  dev.TwitchId,
	})
	s.Error(err) //should have membership not found error

	// assert Owner is intact/uneffected
	ownerNewMemb, err := s.RBAC.GetMembership(s.AdminCtx, &rbacrpc.GetMembershipRequest{
		CompanyId: company.Id,
		TwitchId:  owner.TwitchId,
	})
	s.NoError(err)
	s.Equal("Owner", ownerNewMemb.Role)

	// Verify no other users exist on the new company
	usersRespAfter, err := s.RBAC.GetUsersByCompanyId(s.Ctx, &rbacrpc.GetUsersByCompanyIdRequest{Id: company.Id})
	s.NoError(err)
	s.Equal(uint64(1), usersRespAfter.Total)

	counts = countMemberships(usersRespAfter.Memberships)
	s.Equal(counts.Owners, 1)
	s.Equal(counts.Admins, 0)
	s.Equal(counts.Managers, 0)
	s.Equal(counts.Marketers, 0)
	s.Equal(counts.Developers, 0)
}

func (s *PrivacySuite) Test_CreateCompany_OwnerUserCreated() {
	company := s.createCompany()

	// Get admin created with company
	usersResp, err := s.RBAC.GetUsersByCompanyId(s.Ctx, &rbacrpc.GetUsersByCompanyIdRequest{Id: company.Id})
	s.NoError(err)

	counts := countMemberships(usersResp.Memberships)
	s.Equal(counts.Owners, 1)
	s.Equal(counts.Admins, 0)
	s.Equal(counts.Managers, 0)
	s.Equal(counts.Marketers, 0)
	s.Equal(counts.Developers, 0)
}

func (s *PrivacySuite) Test_AddUsersToCompany_ForAllRoles() {
	company := s.createCompany()
	s.createMembershipsForAllRoles(company.Id)

	counts := s.membershipRoleCounts(company.Id)
	s.Equal(counts.Owners, 1)
	s.Equal(counts.Admins, 1)
	s.Equal(counts.Managers, 1)
	s.Equal(counts.Marketers, 1)
	s.Equal(counts.Developers, 1)
}

func (s *PrivacySuite) Test_AddUsersToCompany_WithLowercaseRoles() {
	company := s.createCompany()

	resp, err := s.RBAC.ListAllRoles(s.Ctx, &rbacrpc.Empty{})
	s.NoError(err)
	for _, role := range resp.Roles {
		if role == "Owner" {
			continue
		}

		s.mustCreateMembership(membershipParams{
			CompanyID: company.Id,
			Role:      strings.ToLower(role),
		})
	}

	counts := s.membershipRoleCounts(company.Id)
	s.Equal(counts.Owners, 1)
	s.Equal(counts.Admins, 1)
	s.Equal(counts.Managers, 1)
	s.Equal(counts.Marketers, 1)
	s.Equal(counts.Developers, 1)

	developer := s.membershipWithRole(company.Id, "Developer")

	// Update role using lowercase notation
	_, err = s.RBAC.UpdateUserRole(s.AdminCtx, &rbacrpc.UpdateUserRoleRequest{
		TwitchId:  developer.TwitchId,
		Role:      "developer", // no change but shouldn't fail
		CompanyId: company.Id,
	})
	s.NoError(err)
}

func (s *PrivacySuite) Test_AddUsersToCompany_ActivityLogs() {
	company := s.createCompany()
	numRolesCreated := s.createMembershipsForAllRoles(company.Id)

	checkCreatedUserActivityLogs := func(users []*rbacrpc.Membership) {
		for _, user := range users {
			s.checkAuthedEntityActionsLen(user.TwitchId, "UserRole", WhitelistAdminTwitchId, company.Id, 1)
		}
	}

	checkCreatedUserActivityLogs(s.membershipsWithRole(company.Id, "Administrator"))
	checkCreatedUserActivityLogs(s.membershipsWithRole(company.Id, "Manager"))
	checkCreatedUserActivityLogs(s.membershipsWithRole(company.Id, "Marketer"))
	checkCreatedUserActivityLogs(s.membershipsWithRole(company.Id, "Developer"))
	s.checkAuthedCompanyActionsLen(company.Id, WhitelistAdminTwitchId, numRolesCreated+7)
}
