package e2e

import (
	"strconv"

	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/twitchtv/twirp"
)

func (s *CompanySuite) Test_RemoveCompanyMembership() {
	company := s.createCompany()
	adminUser := s.mustCreateMembership(membershipParams{CompanyID: company.Id, Role: "Administrator"})

	newUser := s.mustCreateMembership(membershipParams{
		CompanyID:          company.Id,
		Role:               "Developer",
		RequestingTwitchID: adminUser.TwitchId,
	})

	// Random users cannot delete users in this organization
	_, err := s.RBAC.RemoveCompanyMembership(s.Ctx, &rbacrpc.RemoveCompanyMembershipRequest{
		CompanyId:          company.Id,
		TwitchId:           newUser.TwitchId,
		RequestingTwitchId: randomTwitchID(),
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)

	// Try to remove a random user that does not exist in rbac fails
	_, err = s.RBAC.RemoveCompanyMembership(s.Ctx, &rbacrpc.RemoveCompanyMembershipRequest{
		CompanyId:          company.Id,
		TwitchId:           randomTwitchID(),
		RequestingTwitchId: adminUser.TwitchId,
	})
	s.EqualErrorCode(err, twirp.NotFound)

	// Admin user can remove other users
	_, err = s.RBAC.RemoveCompanyMembership(s.Ctx, &rbacrpc.RemoveCompanyMembershipRequest{
		CompanyId:          company.Id,
		TwitchId:           newUser.TwitchId,
		RequestingTwitchId: adminUser.TwitchId,
	})
	s.NoError(err)
}

func (s *CompanySuite) Test_RemoveCompanyMembership_RemoveOneself() {
	// Test users removing their own memberships from companies.
	// This test is only for roles that can remove themselves, others like "Owner" can never leave.
	company := s.createCompany()

	for _, role := range []string{"Administrator", "Marketer", "Manager", "Developer"} {
		twitchID := randomTwitchID()
		s.mustCreateMembership(membershipParams{
			Role:      role,
			CompanyID: company.Id,
			TwitchID:  twitchID,
		})

		_, err := s.RBAC.RemoveCompanyMembership(s.Ctx, &rbacrpc.RemoveCompanyMembershipRequest{
			CompanyId:          company.Id,
			RequestingTwitchId: twitchID,
			TwitchId:           twitchID,
		})
		s.NoError(err)
	}
}

func (s *CompanySuite) Test_RemoveCompanyMembership_AssignedBillingManagerCannotBeRemoved() {
	company := s.createCompany()
	ownerMember := s.membershipWithRole(company.Id, "Owner")

	billingMember := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Billing_Manager",
		TwitchID:  randomTwitchIDWithTIMsEnabled(),
	})

	adminMember := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Administrator",
		TwitchID:  randomTwitchIDWithTIMsEnabled(),
	})

	// Add extension resources
	extensionID09 := "valid-client-id-9"  // Client-ID assumed to be valid by the fake Owl client
	extensionID10 := "valid-client-id-10" // Client-ID assumed to be valid by the fake Owl client
	_, err := s.RBAC.CreateResource(s.AdminCtx, &rbacrpc.CreateResourceRequest{
		CompanyId: company.Id,
		Resource:  &rbacrpc.Resource{ExternalId: extensionID09, Type: "extension"},
	})
	s.NoError(err)
	_, err = s.RBAC.CreateResource(s.AdminCtx, &rbacrpc.CreateResourceRequest{
		CompanyId: company.Id,
		Resource:  &rbacrpc.Resource{ExternalId: extensionID10, Type: "extension"},
	})
	s.NoError(err)

	// Assign billing managers to members with roles "Billing_Manager" and "Administrator"
	_, err = s.RBAC.SetExtensionBillingManager(s.AdminCtx, &rbacrpc.SetExtensionBillingManagerRequest{
		BillingManagerTwitchId: billingMember.TwitchId,
		ExtensionClientId:      extensionID09,
	})
	s.NoError(err)
	_, err = s.RBAC.SetExtensionBillingManager(s.AdminCtx, &rbacrpc.SetExtensionBillingManagerRequest{
		BillingManagerTwitchId: adminMember.TwitchId,
		ExtensionClientId:      extensionID10,
	})
	s.NoError(err)

	// Attempt to remove the assigned billing managers should fail
	_, err = s.RBAC.RemoveCompanyMembership(s.Ctx, &rbacrpc.RemoveCompanyMembershipRequest{
		CompanyId:          company.Id,
		TwitchId:           billingMember.TwitchId,
		RequestingTwitchId: ownerMember.TwitchId,
	})
	s.EqualErrorCode(err, twirp.FailedPrecondition)

	_, err = s.RBAC.RemoveCompanyMembership(s.Ctx, &rbacrpc.RemoveCompanyMembershipRequest{
		CompanyId:          company.Id,
		TwitchId:           adminMember.TwitchId,
		RequestingTwitchId: ownerMember.TwitchId,
	})
	s.EqualErrorCode(err, twirp.FailedPrecondition)

	// Cannot remove themselves either
	_, err = s.RBAC.RemoveCompanyMembership(s.Ctx, &rbacrpc.RemoveCompanyMembershipRequest{
		CompanyId:          company.Id,
		TwitchId:           billingMember.TwitchId,
		RequestingTwitchId: billingMember.TwitchId,
	})
	s.EqualErrorCode(err, twirp.FailedPrecondition)

	_, err = s.RBAC.RemoveCompanyMembership(s.Ctx, &rbacrpc.RemoveCompanyMembershipRequest{
		CompanyId:          company.Id,
		TwitchId:           adminMember.TwitchId,
		RequestingTwitchId: adminMember.TwitchId,
	})
	s.EqualErrorCode(err, twirp.FailedPrecondition)
}

func (s *CompanySuite) Test_RemoveCompanyMembership_OwnerUserCannotBeRemoved() {
	company := s.createCompany()
	owner := s.membershipWithRole(company.Id, "Owner")

	_, err := s.RBAC.RemoveCompanyMembership(s.Ctx, &rbacrpc.RemoveCompanyMembershipRequest{
		CompanyId:          company.Id,
		TwitchId:           owner.TwitchId,
		RequestingTwitchId: owner.TwitchId, // remove oneself
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)
}

func (s *CompanySuite) Test_RemoveCompanyMembership_ShadowAccountCannotBeRemoved() {
	company := s.createCompany()
	resp, err := s.RBACAdmin.CreateShadowAccount(s.AdminCtx, &rbacrpc.CreateShadowAccountRequest{
		ClientId:  "client_id",
		CompanyId: company.Id,
		UserInput: []*rbacrpc.CreateShadowAccountUserInput{
			&rbacrpc.CreateShadowAccountUserInput{
				Email: "testemail@gmail.com",
			},
		},
	})
	s.NoError(err)
	s.NotNil(resp)
	s.NotEmpty(resp.Records)
	s.Equal(1, len(resp.Records))

	twitchIDUint, err := strconv.ParseUint(resp.Records[0].TwitchId, 10, 64)
	s.NoError(err)

	adminMember := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Administrator",
		TwitchID:  randomTwitchIDWithTIMsEnabled(),
	})

	// test removal
	_, err = s.RBAC.RemoveCompanyMembership(s.Ctx, &rbacrpc.RemoveCompanyMembershipRequest{
		CompanyId:          company.Id,
		TwitchId:           strconv.FormatUint(twitchIDUint, 10),
		RequestingTwitchId: adminMember.TwitchId,
	})
	s.EqualErrorCode(err, twirp.PermissionDenied)
}
