#!/bin/bash

set -e

# This script keeps commands used in docker-compose.yml and e2e/ci.sh in one place. This is helpful for ensuring the same file names, flags, etc are used across containers and files. When used for CI, this script expects to be run within docker.

## Helpers. Must be defined before they are used

function log() {
	echo ">>> $@"
}

function logWithCoverage() {
	if [ -z "$NO_COVERAGE" ]; then
		COVERAGE_SUFFIX="with coverage"
	else
		COVERAGE_SUFFIX="without coverage"
	fi
	log "$@ ${COVERAGE_SUFFIX}"
}

function logCoverageFromFile() {
	TEST_TYPE=$1
	COVERAGE_FILE=$2
	go tool cover -func=${COVERAGE_FILE} | grep "total:" | awk '{print $3}' | xargs -I cov_pct echo ">>> ${TEST_TYPE} coverage: cov_pct"
}

function ensureCoverFile() {
	COVER_FILE=$1

	if [ ! -f "${COVER_FILE}" ]; then
		log "missing cover file: ${COVER_FILE}"
		exit 1
	fi
}

## Constants

COVERMODE="atomic"
COV_FILE_PREFIX="coverfile"
SERVER_COVER_FILE=${COV_FILE_PREFIX}_e2e_server.out
CLIENT_COVER_FILE=${COV_FILE_PREFIX}_e2e_client.out

## Parameters -- passed via arguments or environment variables

# NO_COVERAGE: Whether or not to run tests with coverage.
if [ -z "$COVERAGE" ]; then
	export NO_COVERAGE='on'
fi

# CMD: Switch statement to execute
CMD=$1

case $CMD in
	# Ensures tests build
	e2e_tests)
		log "building e2e tests"
		go test -short ./e2e
		;;
	# Builds binary to be tested
	compile)
		if [ -z "$NO_COVERAGE" ]; then
			log "building test binary"
			rm -f devsite-rbac.test
			go test -c -test.covermode ${COVERMODE} -coverpkg ./...
		else
			log "building binary"
			rm -f devsite-rbac
			go build
		fi
		# clean up test files
		rm -f ${CLIENT_COVER_FILE}
		rm -f ${SERVER_COVER_FILE}
		;;
	# runs the client for the e2e tests. the service is expected to already be running
	test)
		logWithCoverage "running e2e tests"
		if [ -z "$NO_COVERAGE" ]; then
			go test -v ./e2e -covermode ${COVERMODE} -coverpkg ./... -coverprofile ${CLIENT_COVER_FILE}
		else
			go test -v ./e2e
		fi
		;;
	# runs the service for the e2e tests.
	service)
		logWithCoverage "running e2e service"
		## The following trap code comes from https://medium.com/@gchudnov/trapping-signals-in-docker-containers-7a57fdda7d86
		# Since we dynamically run the command, bash receives the signal first. We have to pass it to the running process.
		pid=0
		term_handler() {
			if [ $pid -ne 0 ]; then
				log "killing e2e service"
				kill -SIGTERM "$pid"
				log "waiting for e2e service"
				wait "$pid"
				log "waited for e2e service"
			fi
			exit 0;
		}

		trap 'kill ${!}; term_handler' SIGTERM

		if [ -z "$NO_COVERAGE" ]; then
			./devsite-rbac.test -test.coverprofile ${SERVER_COVER_FILE} &
		else
			./devsite-rbac &
		fi
		pid="$!"

		# wait forever
		while true
		do
			tail -f /dev/null & wait ${!}
		done
		log "ran for e2e service"
		;;
	# combines client and service coverage into one file
	collect)
		if [ -z "$NO_COVERAGE" ]; then
			if [ ! -z "$E2E_DEBUG" ]; then
				log "debug mode: skipping test coverage collection"
			else
				# Collect server and client coverage into one file
				ensureCoverFile ${CLIENT_COVER_FILE}
				ensureCoverFile ${SERVER_COVER_FILE}
				echo "mode: ${COVERMODE}" > ./${COV_FILE_PREFIX}_e2e.out
				# remove generated code from code coverage & combine
				tail -q -n +2 ${COV_FILE_PREFIX}_e2e_* | grep -Ev ".twirp.go|.pb.go|noop.go" >> ${COV_FILE_PREFIX}_e2e.out
				logCoverageFromFile e2e ${COV_FILE_PREFIX}_e2e.out
			fi
		else
			log "skipping test coverage collection"
		fi
		;;
	# runs unit tests
	unittest)
		logWithCoverage "running unittests"
		if [ -z "$NO_COVERAGE" ]; then
			go test \
				-short ./... \
				-cover -covermode ${COVERMODE} \
				-coverpkg ./... \
				-coverprofile ${COV_FILE_PREFIX}_unit_unfiltered.out
			# remove generated code from code coverage
			cat ${COV_FILE_PREFIX}_unit_unfiltered.out | grep -Ev "test/|pb.go|twirp.go|noop.go|fake.go|fake_|.gen.go|e2e/" > ${COV_FILE_PREFIX}.out
			logCoverageFromFile unittest ${COV_FILE_PREFIX}.out
		else
			go test -race -short ./...
		fi
		;;
	*)
		echo "unknown command given to e2e/run.sh: $CMD"
		exit 1
esac
