package e2e

import (
	"fmt"

	"code.justin.tv/devrel/devsite-rbac/clients/owlcli"
	"code.justin.tv/devrel/devsite-rbac/models/permissions"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/twitchtv/twirp"
)

func (s *CompanySuite) Test_Validate_CompanyPermissions() {
	company := s.createCompany()

	addUser := &rbacrpc.ValidateQuery{
		Permission:   permissions.AddUser,
		ResourceId:   company.Id,
		ResourceType: permissions.Company,
	}
	showCompanyMembers := &rbacrpc.ValidateQuery{
		Permission:   permissions.ShowCompanyMembers,
		ResourceId:   company.Id,
		ResourceType: permissions.Company,
	}
	removeUser := &rbacrpc.ValidateQuery{
		Permission:   permissions.RemoveUser,
		ResourceId:   company.Id,
		ResourceType: permissions.Company,
	}
	gamesAdd := &rbacrpc.ValidateQuery{
		Permission:   permissions.AddGames,
		ResourceId:   company.Id,
		ResourceType: permissions.Company,
	}
	dropsView := &rbacrpc.ValidateQuery{
		Permission:   permissions.ViewDrops,
		ResourceId:   company.Id,
		ResourceType: permissions.Company,
	}
	dropsCreate := &rbacrpc.ValidateQuery{
		Permission:   permissions.CreateDrops,
		ResourceId:   company.Id,
		ResourceType: permissions.Company,
	}
	dropsManage := &rbacrpc.ValidateQuery{
		Permission:   permissions.ManageDrops,
		ResourceId:   company.Id,
		ResourceType: permissions.Company,
	}
	dropsDelete := &rbacrpc.ValidateQuery{
		Permission:   permissions.DeleteDrops,
		ResourceId:   company.Id,
		ResourceType: permissions.Company,
	}
	insightsOnePager := &rbacrpc.ValidateQuery{
		Permission:   permissions.InsightsOnePagerView,
		ResourceId:   company.Id,
		ResourceType: permissions.Company,
	}
	viewGame := &rbacrpc.ValidateQuery{
		Permission:   permissions.ViewGame,
		ResourceId:   company.Id,
		ResourceType: permissions.Company,
	}

	s.checkCompanyRolePermissions(company.Id, []permissionCase{
		{"Owner", []*rbacrpc.ValidateQuery{
			addUser, removeUser, gamesAdd, dropsView, dropsManage, dropsCreate, dropsDelete, insightsOnePager, showCompanyMembers, viewGame,
		}, nil},

		// Administrators can manage apps, games, drops, and adding and removing users
		{"Administrator", []*rbacrpc.ValidateQuery{
			addUser, removeUser, gamesAdd, dropsView, dropsManage, dropsCreate, dropsDelete, insightsOnePager, showCompanyMembers, viewGame,
		}, nil},

		{"Billing_Manager", []*rbacrpc.ValidateQuery{
			addUser, removeUser, gamesAdd, dropsView, dropsManage, dropsCreate, dropsDelete, insightsOnePager, showCompanyMembers, viewGame,
		}, []*rbacrpc.ValidateQuery{}},

		// Managers can manage apps, games, drops, and add users, but cannot remove users
		{"Manager", []*rbacrpc.ValidateQuery{
			addUser, gamesAdd, dropsView, dropsManage, dropsCreate, dropsDelete, insightsOnePager, showCompanyMembers, viewGame,
		}, []*rbacrpc.ValidateQuery{
			removeUser,
		}},

		// Marketers can manage games, drops, and box art for a company
		{"Marketer", []*rbacrpc.ValidateQuery{
			gamesAdd, dropsView, dropsManage, dropsCreate, dropsDelete, insightsOnePager, showCompanyMembers, viewGame,
		}, []*rbacrpc.ValidateQuery{
			addUser, removeUser,
		}},

		// Developers can set and configure drops for a company
		{"Developer", []*rbacrpc.ValidateQuery{
			dropsView, dropsManage, dropsCreate, dropsDelete, insightsOnePager, showCompanyMembers, viewGame,
		}, []*rbacrpc.ValidateQuery{
			addUser, removeUser, gamesAdd,
		}},

		// Random user should have permission to nothing in company
		{randomRole, nil, []*rbacrpc.ValidateQuery{
			addUser, removeUser, gamesAdd, dropsView, dropsManage, dropsCreate, dropsDelete, insightsOnePager, showCompanyMembers, viewGame,
		}},
	}, nil)
}

func (s *CompanySuite) Test_Validate_UserPermissions() {
	company := s.createCompany()
	s.createMembershipsForAllRoles(company.Id)

	ownerUser := s.membershipWithRole(company.Id, "Owner")
	adminUser := s.membershipWithRole(company.Id, "Administrator")
	billingManager := s.membershipWithRole(company.Id, "Billing_Manager")
	managerUser := s.membershipWithRole(company.Id, "Manager")
	marketerUser := s.membershipWithRole(company.Id, "Marketer")
	developerUser := s.membershipWithRole(company.Id, "Developer")

	userQueries := func(permission string, members ...*rbacrpc.Membership) []*rbacrpc.ValidateQuery {
		queries := make([]*rbacrpc.ValidateQuery, len(members))
		for i, member := range members {
			queries[i] = &rbacrpc.ValidateQuery{
				ResourceType: permissions.User,
				ResourceId:   member.TwitchId,
				Permission:   permission,
			}
		}
		return queries
	}

	paymentOnboarding := userQueries(permissions.ViewPaymentOnboarding, ownerUser, adminUser, billingManager, managerUser, marketerUser, developerUser)
	// allow owners to view onboarding state of any users
	paymentOnboarding = append(paymentOnboarding, &rbacrpc.ValidateQuery{
		ResourceType: permissions.User,
		ResourceId:   randomTwitchID(),
		Permission:   permissions.ViewPaymentOnboarding,
	})

	s.checkCompanyRolePermissions(company.Id, []permissionCase{
		{"Owner", paymentOnboarding, nil},
		{"Administrator", nil, paymentOnboarding},
		{"Billing_Manager", nil, paymentOnboarding},
		{"Manager", nil, paymentOnboarding},
		{"Marketer", nil, paymentOnboarding},
		{"Developer", nil, paymentOnboarding},
		{randomRole, nil, paymentOnboarding},
	}, &permissionOpts{
		SkipCreatingMemberships: true,
	})
}

func (s *CompanySuite) Test_Validate_ViewPaymentOnboarding_ViennaWhitelistedUsers() {
	resp, err := s.RBAC.ValidateByTwitchID(s.Ctx, &rbacrpc.ValidateQuery{
		UserId:       WhitelistAdminTwitchId, // whitelisted user
		ResourceId:   randomString(10),
		ResourceType: permissions.User,
		Permission:   permissions.ViewPaymentOnboarding,
	})
	s.NoError(err)
	s.True(resp.Valid, "Vienna Whitelisted User should be able to see ViewPaymentOnboarding of any user")
}

func (s *CompanySuite) Test_Validate_ResourceNotFound() {
	company := s.createCompany()
	owner := s.membershipWithRole(company.Id, "Owner")

	_, err := s.RBAC.ValidateByTwitchID(s.Ctx, &rbacrpc.ValidateQuery{
		UserId:       owner.TwitchId,
		ResourceId:   randomString(10),
		ResourceType: randomString(10),
		Permission:   randomString(10),
	})
	s.EqualTwirp(err, twirp.NotFound, "resource")
}

func (s *CompanySuite) Test_Validate_ResourceNotFound_UserNoRoles() {
	_, err := s.RBAC.ValidateByTwitchID(s.Ctx, &rbacrpc.ValidateQuery{
		UserId:       randomString(10), // user doesn't have roles
		ResourceId:   randomString(10),
		ResourceType: randomString(10),
		Permission:   randomString(10),
	})
	s.EqualTwirp(err, twirp.NotFound, "resource")
}

func (s *CompanySuite) Test_Validate_GameResources() {
	company := s.createCompany()
	game := s.createOnboardedGame(company.Id)

	// Verify box art edit
	// Verify analytics view
	boxArtEdit := &rbacrpc.ValidateQuery{
		Permission:   permissions.GameBoxArtEdit,
		ResourceId:   fmt.Sprintf("%d", game.Id),
		ResourceType: permissions.Game,
	}

	analyticsView := &rbacrpc.ValidateQuery{
		Permission:   permissions.GameAnalyticsView,
		ResourceId:   fmt.Sprintf("%d", game.Id),
		ResourceType: permissions.Game,
	}

	s.checkCompanyRolePermissions(company.Id, []permissionCase{
		{"Owner", []*rbacrpc.ValidateQuery{
			boxArtEdit, analyticsView,
		}, nil},

		// Administrators can manage apps, games, drops, and adding and removing users
		{"Administrator", []*rbacrpc.ValidateQuery{
			boxArtEdit, analyticsView,
		}, nil},

		{"Billing_Manager", []*rbacrpc.ValidateQuery{
			boxArtEdit, analyticsView,
		}, nil},

		// Managers can manage apps, games, drops, and add users, but cannot remove users
		{"Manager", []*rbacrpc.ValidateQuery{
			boxArtEdit, analyticsView,
		}, nil},

		// Marketers can manage games, drops, and box art for a company
		{"Marketer", []*rbacrpc.ValidateQuery{
			boxArtEdit, analyticsView,
		}, nil},

		// Developers can set and configure drops for a company
		{"Developer", []*rbacrpc.ValidateQuery{
			analyticsView,
		}, []*rbacrpc.ValidateQuery{
			boxArtEdit,
		}},

		// Random user shouldn't have either permissions
		{randomRole, nil, []*rbacrpc.ValidateQuery{
			boxArtEdit, analyticsView,
		}},
	}, nil)
}

func (s *CompanySuite) Test_Validate_Extensions() {
	company := s.createCompany()

	extensionID := randomString(16)

	_, err := s.RBAC.CreateResource(s.Ctx, &rbacrpc.CreateResourceRequest{
		CompanyId: company.Id,
		Resource: &rbacrpc.Resource{
			ExternalId: extensionID,
			Type:       permissions.Extension,
		},
	})
	s.NoError(err)

	createExtensions := &rbacrpc.ValidateQuery{
		Permission:   permissions.CreateExtensions,
		ResourceId:   extensionID,
		ResourceType: permissions.Extension,
	}

	manageExtensions := &rbacrpc.ValidateQuery{
		Permission:   permissions.ManageExtensions,
		ResourceId:   extensionID,
		ResourceType: permissions.Extension,
	}

	monetizeExtensions := &rbacrpc.ValidateQuery{
		Permission:   permissions.MonetizeExtensions,
		ResourceId:   extensionID,
		ResourceType: permissions.Extension,
	}

	viewInsights := &rbacrpc.ValidateQuery{
		Permission:   permissions.ViewExtensionInsights,
		ResourceId:   extensionID,
		ResourceType: permissions.Extension,
	}

	s.checkCompanyRolePermissions(company.Id, []permissionCase{
		{"Owner", []*rbacrpc.ValidateQuery{
			createExtensions, manageExtensions, monetizeExtensions, viewInsights,
		}, nil},

		// Administrators can manage apps, games, drops, and adding and removing users
		{"Administrator", []*rbacrpc.ValidateQuery{
			createExtensions, manageExtensions, monetizeExtensions, viewInsights,
		}, nil},

		{"Billing_Manager", []*rbacrpc.ValidateQuery{
			createExtensions, manageExtensions, monetizeExtensions, viewInsights,
		}, nil},

		// Managers can manage apps, games, drops, and add users, but cannot remove users
		{"Manager", []*rbacrpc.ValidateQuery{
			createExtensions, manageExtensions, monetizeExtensions, viewInsights,
		}, nil},

		// Marketers can manage games, drops, and box art for a company
		{"Marketer", []*rbacrpc.ValidateQuery{
			createExtensions, manageExtensions, monetizeExtensions, viewInsights,
		}, nil},

		// Developers can set and configure drops for a company
		{"Developer", []*rbacrpc.ValidateQuery{
			createExtensions, manageExtensions, monetizeExtensions, viewInsights,
		}, nil},

		// Random user shouldn't have either permissions
		{randomRole, nil, []*rbacrpc.ValidateQuery{
			createExtensions, manageExtensions, viewInsights, monetizeExtensions,
		}},
	}, nil)
}

func (s *CompanySuite) Test_ValidateExtensionAccess() {
	company := s.createCompany()
	devUser := s.mustCreateMembership(membershipParams{
		CompanyID: company.Id,
		Role:      "Developer",
	})

	// Org-owned extension
	orgExtensionID := randomString(16)
	_, err := s.RBAC.CreateResource(s.Ctx, &rbacrpc.CreateResourceRequest{
		CompanyId: company.Id,
		Resource: &rbacrpc.Resource{
			ExternalId: orgExtensionID,
			Type:       permissions.Extension,
		},
	})
	s.NoError(err)

	// Developer in same org has access
	resp, err := s.RBAC.ValidateExtensionAccess(s.Ctx, &rbacrpc.ValidateExtensionAccessRequest{
		UserId:      devUser.TwitchId,
		ExtensionId: orgExtensionID,
		Permission:  permissions.ViewExtensionInsights,
	})
	s.NoError(err)
	s.Equal(true, resp.Valid)
	s.Equal(true, resp.OrgOwned)

	// Another developer outside of the org doesn't have access
	resp, err = s.RBAC.ValidateExtensionAccess(s.Ctx, &rbacrpc.ValidateExtensionAccessRequest{
		UserId:      randomTwitchID(), // another user
		ExtensionId: orgExtensionID,
		Permission:  permissions.ViewExtensionInsights,
	})
	s.NoError(err)
	s.Equal(false, resp.Valid)
	s.Equal(true, resp.OrgOwned)

	// Org-less personal extension is checked in Owl
	personalExtensionID := "client-id-00001"
	validOwnerID := owlcli.FakeOwnerIDForClientID(personalExtensionID)
	resp, err = s.RBAC.ValidateExtensionAccess(s.Ctx, &rbacrpc.ValidateExtensionAccessRequest{
		UserId:      validOwnerID,
		ExtensionId: personalExtensionID,
		Permission:  permissions.ViewExtensionInsights,
	})
	s.NoError(err)
	s.Equal(true, resp.Valid)
	s.Equal(false, resp.OrgOwned)

	// Org-less personal extension needs to match the owner with the current user
	resp, err = s.RBAC.ValidateExtensionAccess(s.Ctx, &rbacrpc.ValidateExtensionAccessRequest{
		UserId:      devUser.TwitchId, // not the owner
		ExtensionId: personalExtensionID,
		Permission:  permissions.ViewExtensionInsights,
	})
	s.NoError(err)
	s.Equal(false, resp.Valid)
	s.Equal(false, resp.OrgOwned)

	// If the extension id is invalid, it returns NotFound
	_, err = s.RBAC.ValidateExtensionAccess(s.Ctx, &rbacrpc.ValidateExtensionAccessRequest{
		UserId:      devUser.TwitchId,
		ExtensionId: "invalid-id-triggers-owl-ErrInvalidClientID",
		Permission:  permissions.ViewExtensionInsights,
	})
	s.EqualTwirp(err, twirp.NotFound, "extension not found")
}
