import { PROD_ACCOUNT_ID, DEV_GITHUB_URL, PROD_GITHUB_URL, PROD_ECR_URL, DEV_ECR_URL } from './consts';
import { StackProps, Stack, Construct } from '@aws-cdk/core';
import { IVpc, SecurityGroup, Port } from '@aws-cdk/aws-ec2';
import { IBaseService } from '@aws-cdk/aws-ecs';
import { IBucket, Bucket } from '@aws-cdk/aws-s3';
import { Trail, ReadWriteType } from '@aws-cdk/aws-cloudtrail';
import { Pipeline, Artifact } from '@aws-cdk/aws-codepipeline';
import { BuildSpec, PipelineProject, Source, Project, Artifacts } from '@aws-cdk/aws-codebuild';
import {
  CodeBuildAction,
  S3SourceAction,
  S3Trigger,
  EcsDeployAction,
  ManualApprovalAction,
} from '@aws-cdk/aws-codepipeline-actions';
import { PolicyStatement, Effect } from '@aws-cdk/aws-iam';
interface DeployStackProps extends StackProps {
  vpc: IVpc;
  deployService: IBaseService;
  githubSecurityGroup: SecurityGroup;
}

export class DeployStack extends Stack {
  PIPELINES_BUCKET: IBucket;
  PROPS: DeployStackProps;
  PROD: boolean;

  constructor(scope: Construct, id: string, props: DeployStackProps) {
    super(scope, id, props);

    this.PROPS = props;

    this.PROD = props.env!.account === PROD_ACCOUNT_ID;

    this.PIPELINES_BUCKET = new Bucket(this, 'RBACPipelinesBucket', {
      versioned: true,
      publicReadAccess: false,
    });

    // We need to trail changes to the ZIP, we add that to S3 now
    const trail = new Trail(this, 'CloudTrail');
    trail.addS3EventSelector([this.PIPELINES_BUCKET.arnForObjects('rbac-codepipelines/RBACSource.zip')], {
      readWriteType: ReadWriteType.WRITE_ONLY,
    });
    trail.addS3EventSelector([this.PIPELINES_BUCKET.arnForObjects('rbac-codepipelines-deploy/RBACSource.zip')], {
      readWriteType: ReadWriteType.WRITE_ONLY,
    });

    const testProject = new PipelineProject(this, 'RbacTest', {
      description: 'Test Process for RBAC',
      projectName: 'RBAC-Test',
      vpc: props.vpc,
      environment: {
        privileged: true,
      },
      buildSpec: BuildSpec.fromObject({
        version: '0.2',
        phases: {
          build: {
            commands: ['echo Running Test suite', 'make ci'],
          },
        },
      }),
    });

    const buildProject = new PipelineProject(this, 'RbacBuild', {
      description: 'Build/Upload to ECR for RBAC',
      projectName: 'RBAC-Build',
      vpc: props.vpc,
      environment: {
        privileged: true,
      },
      buildSpec: BuildSpec.fromObject({
        version: '0.2',
        phases: {
          pre_build: {
            commands: [
              'echo Logging into ECR...',
              '$(aws ecr get-login --region $AWS_DEFAULT_REGION --no-include-email)',
              'REPOSITORY_URI=' + (this.PROD ? PROD_ECR_URL : DEV_ECR_URL),
              'echo Getting Commit Hash',
              'COMMIT_HASH=$(cat gitHash.txt)',
              'echo /////////////////////////////////////////////////////////////////',
              'echo /////////////////////////////////////////////////////////////////',
              'echo /////////////////////////////////////////////////////////////////',
              'echo ///////////////////// Deploying $COMMIT_HASH //////////////////////////',
              'echo /////////////////////////////////////////////////////////////////',
              'echo /////////////////////////////////////////////////////////////////',
              'echo /////////////////////////////////////////////////////////////////',
              // tslint:disable-next-line: no-invalid-template-strings
              'IMAGE_TAG=${COMMIT_HASH:=latest}',
            ],
          },
          build: {
            commands: [
              'echo Build started on `date`',
              'echo Building the Docker image...',
              'docker build -t $REPOSITORY_URI:latest .',
              'docker tag $REPOSITORY_URI:latest $REPOSITORY_URI:$IMAGE_TAG',
            ],
          },
          post_build: {
            commands: [
              'echo Build completed on `date`',
              'echo Docker URL $REPOSITORY_URI:$IMAGE_TAG',
              'echo Pushing the Docker images...',
              'docker push $REPOSITORY_URI:latest',
              'docker push $REPOSITORY_URI:$IMAGE_TAG',
              'echo Writing image definitions file...',
              'printf \'[{"name":"rbac","imageUri":"%s"}]\' $REPOSITORY_URI:$IMAGE_TAG > imagedefinitions.json',
            ],
          },
        },
        artifacts: {
          files: 'imagedefinitions.json', // You CANNOT change this, its a known value to Pipelines
        },
      }),
    });

    buildProject.addToRolePolicy(
      new PolicyStatement({
        actions: ['ecr:*'],
        effect: Effect.ALLOW,
        resources: ['*'],
      })
    );

    if (!this.PROD) this.SetupTestingPipeline(testProject, buildProject);
    this.SetupDeployPipeline(testProject, buildProject);
  }

  private SetupTestingPipeline(testProject: PipelineProject, buildProject: PipelineProject) {
    const gitHubSource = Source.gitHubEnterprise({
      httpsCloneUrl: this.PROD ? PROD_GITHUB_URL : DEV_GITHUB_URL,
      // webhook: true,  // NOTE: Webhooks currently don't work with Github Enterprise, you need to recreate this in the console if you update the codebuild object
    });

    // In order to provide GHE source to Pipelines we must upload it as an S3 artifact first
    const rbacTestSourceProject = new Project(this, 'RBACTestSource', {
      source: gitHubSource,
      description: 'Fetches RBACs source from Github Enterprise and uploads it to S3 for Pipeline to consume',
      projectName: 'RBAC-Test-Source',
      buildSpec: BuildSpec.fromObject({
        version: '0.2',
        phases: {
          install: {
            'runtime-versions': {
              docker: 18,
            },
          },
          pre_build: {
            commands: ['COMMIT_HASH=$(echo $CODEBUILD_RESOLVED_SOURCE_VERSION | cut -c 1-7)'],
          },
          build: {
            commands: [
              'echo Uploading Artifacts to S3',
              'echo $CODEBUILD_WEBHOOK_HEAD_REF > branch.txt',
              'echo $(echo $CODEBUILD_RESOLVED_SOURCE_VERSION | cut -c 1-7) > gitHash.txt',
            ],
          },
        },
        artifacts: {
          files: ['**/*'],
        },
      }),
      artifacts: Artifacts.s3({
        bucket: this.PIPELINES_BUCKET,
        name: 'RBACSource.zip',
        path: 'rbac-codepipelines',
        includeBuildId: false,
      }),
      vpc: this.PROPS.vpc,
    });

    this.PROPS.githubSecurityGroup.connections.allowFrom(
      rbacTestSourceProject.connections,
      Port.tcp(443),
      'Allow access to RBAC Test Source CloudBuild'
    );

    const rbacGHETestProject = new Project(this, 'RBAC-GHE-Test', {
      source: gitHubSource,
      description:
        'Fetches RBACs source from Github Enterprise and runs tests/build to verify status. Then reports that status to GHE.',
      projectName: 'RBAC-GHE-Test',
      buildSpec: BuildSpec.fromObject({
        version: '0.2',
        phases: {
          install: {
            'runtime-versions': {
              docker: 18,
            },
          },
          pre_build: {
            commands: ['COMMIT_HASH=$(echo $CODEBUILD_RESOLVED_SOURCE_VERSION | cut -c 1-7)'],
          },
          build: {
            commands: ['echo Running CI', 'make ci', 'echo Running DockerBuild', 'docker build .'],
          },
        },
      }),
      vpc: this.PROPS.vpc,
      environment: {
        privileged: true,
      },
    });

    this.PROPS.githubSecurityGroup.connections.allowFrom(
      rbacGHETestProject.connections,
      Port.tcp(443),
      'Allow access to RBAC GHE Testing CloudBuild'
    );

    const pipeline = new Pipeline(this, 'RbacTestPipeline', {
      pipelineName: 'Rbac-Test-' + (this.PROD ? 'Production' : 'Staging'),
    });

    const sourceOutput = new Artifact();
    pipeline.addStage({
      stageName: 'Source',
      actions: [
        new S3SourceAction({
          actionName: 'Fetch_Source_from_S3',
          bucket: this.PIPELINES_BUCKET,
          bucketKey: 'rbac-codepipelines/RBACSource.zip',
          output: sourceOutput,
          trigger: S3Trigger.EVENTS,
        }),
      ],
    });

    pipeline.addStage({
      stageName: 'Test',
      actions: [
        new CodeBuildAction({
          actionName: 'Run_Tests',
          project: testProject,
          input: sourceOutput,
        }),
      ],
    });

    const imageDefs = new Artifact();
    pipeline.addStage({
      stageName: 'Build',
      actions: [
        new CodeBuildAction({
          actionName: 'Build_Docker',
          project: buildProject,
          input: sourceOutput,
          outputs: [imageDefs],
        }),
      ],
    });
  }

  private SetupDeployPipeline(testProject: PipelineProject, buildProject: PipelineProject) {
    const gitHubSource = Source.gitHubEnterprise({
      httpsCloneUrl: this.PROD ? PROD_GITHUB_URL : DEV_GITHUB_URL,
      branchOrRef: this.PROD ? 'master' : 'develop',
      // webhook: true,  // NOTE: Webhooks currently don't work with Github Enterprise, you need to recreate this in the console if you update the codebuild object
    });

    // In order to provide GHE source to Pipelines we must upload it as an S3 artifact first
    // Branch filter (set with the webhooks) ^refs/heads/master$ / ^refs/heads/develop$
    // PUSH events only
    // Log into Isengard and then https://us-west-2.console.aws.amazon.com/codesuite/codebuild/projects/RBAC-Deploy-Source/edit/source?region=us-west-2
    const rbacDeployProject = new Project(this, 'RBACDeploySource', {
      source: gitHubSource,
      description: 'Fetches RBACs source from Github Enterprise and uploads it to S3 for Pipeline to consume',
      projectName: 'RBAC-Deploy-Source',
      buildSpec: BuildSpec.fromObject({
        version: '0.2',
        phases: {
          install: {
            'runtime-versions': {
              docker: 18,
            },
          },
          build: {
            commands: [
              'echo Uploading Artifacts to S3',
              'echo $CODEBUILD_WEBHOOK_HEAD_REF > branch.txt',
              'echo $(echo $CODEBUILD_RESOLVED_SOURCE_VERSION | cut -c 1-7) > gitHash.txt',
            ],
          },
        },
        artifacts: {
          files: ['**/*'],
        },
      }),
      artifacts: Artifacts.s3({
        bucket: this.PIPELINES_BUCKET,
        name: 'RBACSource.zip',
        path: 'rbac-codepipelines-deploy',
        includeBuildId: false,
      }),
      vpc: this.PROPS.vpc,
    });

    this.PROPS.githubSecurityGroup.connections.allowFrom(
      rbacDeployProject.connections,
      Port.tcp(443),
      'Allow access to RBAC Deploy CloudBuild'
    );

    const pipeline = new Pipeline(this, 'RbacDeployPipeline', {
      pipelineName: 'Rbac-Deploy-' + (this.PROD ? 'Production' : 'Staging'),
    });

    const sourceOutput = new Artifact();
    pipeline.addStage({
      stageName: 'Source',
      actions: [
        new S3SourceAction({
          actionName: 'Fetch_Source_from_S3',
          bucket: this.PIPELINES_BUCKET,
          bucketKey: 'rbac-codepipelines-deploy/RBACSource.zip',
          output: sourceOutput,
          trigger: S3Trigger.EVENTS,
        }),
      ],
    });

    pipeline.addStage({
      stageName: 'Test',
      actions: [
        new CodeBuildAction({
          actionName: 'Run_Tests',
          project: testProject,
          input: sourceOutput,
        }),
      ],
    });

    const imageDefs = new Artifact();
    pipeline.addStage({
      stageName: 'Build',
      actions: [
        new CodeBuildAction({
          actionName: 'Build_Docker',
          project: buildProject,
          input: sourceOutput,
          outputs: [imageDefs],
        }),
      ],
    });

    const deployStage = pipeline.addStage({
      stageName: 'Deploy',
      actions: [
        new EcsDeployAction({
          actionName: 'Deploy_Service',
          input: imageDefs,
          service: this.PROPS.deployService,
          runOrder: 2,
        }),
      ],
    });

    if (this.PROD) {
      deployStage.addAction(
        new ManualApprovalAction({
          actionName: 'Approve',
          runOrder: 1,
        })
      );
    }
  }
}
