import { Rule, Schedule } from '@aws-cdk/aws-events';
import { LambdaFunction } from '@aws-cdk/aws-events-targets';
import { Construct, Duration, Stack, StackProps } from '@aws-cdk/core';
import { DEV_RBAC_A2Z, PROD_ACCOUNT_ID, PROD_RBAC_A2Z } from './consts';
import { RbacStack } from './rbac-stack';
import * as lambda from '@aws-cdk/aws-lambda';
import path = require('path');
import { CommonStack } from './common-stack';
import { IVpc } from '@aws-cdk/aws-ec2';

interface PdmsCronStackProps extends StackProps {
  vpc: IVpc;
  rbacStack: RbacStack;
  commonStack: CommonStack;
}

export class PdmsCronStack extends Stack {
  constructor(scope: Construct, id: string, props: PdmsCronStackProps) {
    super(scope, id, props);
    this.addDependency(props.rbacStack);
    const prod = props.env!.account === PROD_ACCOUNT_ID;

    const fn = new lambda.Function(this, 'PdmsCallRbac', {
      runtime: lambda.Runtime.GO_1_X,
      handler: 'main',
      code: lambda.Code.fromAsset(path.join(__dirname, 'pdms-lambda')),
      environment: {
        RBAC_API_KEY: props.commonStack.pdmsSecret.secretValue.toString(),
        RBAC_URL: `https://${
          prod ? PROD_RBAC_A2Z : DEV_RBAC_A2Z
        }/twirp/code.justin.tv.devrel.rbac.Privacy/AwsCloudwatch`,
      },
	  vpc: props.vpc,
    });

    new Rule(this, 'ScheduleRule', {
      schedule: Schedule.rate(Duration.hours(6)),
      targets: [new LambdaFunction(fn)],
    });
  }
}
