import {
  PROD_ACCOUNT_ID,
  DEV_VPC_ID,
  PROD_VPC_ID,
  PROD_GITHUB_VPCE_ID,
  DEV_GITHUB_VPCE_ID,
  PROD_GITHUB_VPCE_URL,
  DEV_GITHUB_VPCE_URL,
  GITHUB_VPCE_HOSTED_ZONE,
  PROD_INTERNAL_SG_ID,
  DEV_INTERNAL_SG_ID,
} from './consts';
import { SecurityGroup, Vpc, IVpc, ISecurityGroup, InterfaceVpcEndpoint } from '@aws-cdk/aws-ec2';
import { Construct, Stack, StackProps, Tag } from '@aws-cdk/core';
import { PrivateHostedZone, IAliasRecordTarget, RecordTarget, ARecord } from '@aws-cdk/aws-route53';

export class VpcStack extends Stack {
  public readonly vpc: IVpc;
  public readonly githubSecurityGroup: SecurityGroup;
  public readonly generalVpcESecurityGroup: SecurityGroup;
  public readonly twitchInternalSecurityGroup: ISecurityGroup;
  public readonly dartVpcE: InterfaceVpcEndpoint;
  public readonly cartmanVpcE: InterfaceVpcEndpoint;
  public readonly discoveryVpcE: InterfaceVpcEndpoint;
  public readonly owlVpcE: InterfaceVpcEndpoint;
  public readonly userVpcE: InterfaceVpcEndpoint;
  public readonly emsVpcE: InterfaceVpcEndpoint;

  constructor(scope: Construct, name: string, props: StackProps) {
    super(scope, name, props);

    // NOTE
    // Please use TPS to create VPCe/privatelink connections, not this!

    const prod = props.env!.account === PROD_ACCOUNT_ID;

    this.vpc = Vpc.fromLookup(this, 'Vpc', {
      vpcId: prod ? PROD_VPC_ID : DEV_VPC_ID,
    });

    this.githubSecurityGroup = new SecurityGroup(this, 'GithubEnterpriseSG', {
      vpc: this.vpc,
      description: 'Allow access to the Github Enterprises VPCe',
    });

    this.generalVpcESecurityGroup = new SecurityGroup(this, 'GeneralVpcESG', {
      vpc: this.vpc,
      description: 'Allow access to general VPCes in the account',
    });

    this.twitchInternalSecurityGroup = SecurityGroup.fromSecurityGroupId(
      this,
      'TwitchInternalSG',
      prod ? PROD_INTERNAL_SG_ID : DEV_INTERNAL_SG_ID
    );

    const gitAws = new PrivateHostedZone(this, 'GithubEnterpriseHZ', {
      zoneName: 'git.xarth.tv',
      vpc: this.vpc,
    });

    // Leaving this uncommented but it is currently only used for the broken ARecord below
    InterfaceVpcEndpoint.fromInterfaceVpcEndpointAttributes(this, 'GithubEnterpriceVPCe', {
      port: 443,
      vpcEndpointId: prod ? PROD_GITHUB_VPCE_ID : DEV_GITHUB_VPCE_ID,
      securityGroups: [this.githubSecurityGroup],
    });

    // This is the proper way to do it but InterfaceVpcEndpointTarget is abusing interfaces and needs to be fixed
    // new route53.ARecord(this, 'GHEARecord', {
    //   zone: gitAws,
    //   target: route53.RecordTarget.fromAlias(new r53targets.InterfaceVpcEndpointTarget(githubVpcE)),
    // });

    // This is the temporary way to do it
    const aliasTargetForGHE: IAliasRecordTarget = {
      bind: () => {
        return {
          hostedZoneId: GITHUB_VPCE_HOSTED_ZONE,
          dnsName: prod ? PROD_GITHUB_VPCE_URL : DEV_GITHUB_VPCE_URL,
        };
      },
    };

    new ARecord(this, 'GHEARecord', {
      zone: gitAws,
      target: RecordTarget.fromAlias(aliasTargetForGHE),
    });

    // This cannot be used until CDK decides to not hardcode TG protocols -> https://github.com/aws/aws-cdk/blob/16e85df5c3077496d3ebe7c4fa8230514756c027/packages/%40aws-cdk/aws-elasticloadbalancingv2/lib/nlb/network-target-group.ts#L60
    // const rbacVpceNLB = new elbv2.NetworkLoadBalancer(this, 'RbacVpceLB', {
    //   vpc: this.vpc,
    // });

    // // Add a listener on a particular port.
    // rbacVpceNLB
    //   .addListener('RbacVpceLBListener', {
    //     port: 443,
    //     certificates: [elbv2.ListenerCertificate.fromArn(prod ? PROD_CERT_ARN : DEV_CERT_ARN)],
    //     protocol: elbv2.Protocol.TLS,
    //   })
    //   .addTargets('RbacVpceLBListenerTargets', {
    //     port: 443,
    //     targets: [new helpers.EmptyIpTarget()],
    //   });

    //#region VPCe
    this.dartVpcE = new InterfaceVpcEndpoint(this, 'DartVpcEndpoint', {
      vpc: this.vpc,
      service: {
        name: prod
          ? 'com.amazonaws.vpce.us-west-2.vpce-svc-0ce6ec756b1a5acba'
          : 'com.amazonaws.vpce.us-west-2.vpce-svc-00cc5f5ddd42126a0',
        port: 443,
      },
      privateDnsEnabled: false, // They do not provide it...
    });

    Tag.add(this.dartVpcE, 'Name', 'Dart');

    this.cartmanVpcE = new InterfaceVpcEndpoint(this, 'CartmanVpcEndpoint', {
      vpc: this.vpc,
      service: {
        name: prod
          ? 'com.amazonaws.vpce.us-west-2.vpce-svc-03ade4b2e790128c1'
          : 'com.amazonaws.vpce.us-west-2.vpce-svc-0d48ba120d8a41fe2',
        port: 443,
      },
      privateDnsEnabled: false, // They do not provide it...
    });

    Tag.add(this.cartmanVpcE, 'Name', 'Cartman');

    this.discoveryVpcE = new InterfaceVpcEndpoint(this, 'DiscoveryVpcEndpoint', {
      vpc: this.vpc,
      service: {
        name: prod
          ? 'com.amazonaws.vpce.us-west-2.vpce-svc-03580e04235e0da93'
          : 'com.amazonaws.vpce.us-west-2.vpce-svc-02972ec3644c35afa',
        port: 80,
      },
      privateDnsEnabled: false, // They do not provide it...
    });

    Tag.add(this.discoveryVpcE, 'Name', 'Discovery');

    this.owlVpcE = new InterfaceVpcEndpoint(this, 'OwlVpcEndpoint', {
      vpc: this.vpc,
      service: {
        name: prod
          ? 'com.amazonaws.vpce.us-west-2.vpce-svc-08fdcf80669bbcb87'
          : 'com.amazonaws.vpce.us-west-2.vpce-svc-02a7897a39672d1a5',
        port: 443,
      },
      privateDnsEnabled: false, // They do not provide it...
    });

    Tag.add(this.owlVpcE, 'Name', 'Owl');

    this.userVpcE = new InterfaceVpcEndpoint(this, 'UserVpcEndpoint', {
      vpc: this.vpc,
      service: {
        name: prod
          ? 'com.amazonaws.vpce.us-west-2.vpce-svc-044e493c84b7dc984'
          : 'com.amazonaws.vpce.us-west-2.vpce-svc-05dd975d62ccc04fe',
        port: 443,
      },
      privateDnsEnabled: false, // They do not provide it...
    });

    Tag.add(this.userVpcE, 'Name', 'User');

    this.emsVpcE = new InterfaceVpcEndpoint(this, 'EMSVpcEndpoint', {
      vpc: this.vpc,
      service: {
        name: prod
          ? 'com.amazonaws.vpce.us-west-2.vpce-svc-065d9a5722b589fa5'
          : 'com.amazonaws.vpce.us-west-2.vpce-svc-094a32bb0c30b9833',
        port: 443,
      },
      privateDnsEnabled: false, // They do not provide it...
    });

    Tag.add(this.userVpcE, 'Name', 'EMS');
    //#endregion
  }
}
