package errorutil

import (
	"database/sql"
	"reflect"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/foundation/twitchclient"
	"github.com/lib/pq"
	uuid "github.com/satori/go.uuid"
	"github.com/twitchtv/twirp"
)

// IsErrNoRows checks if the error was returned by a QueryRow with no retults.
// This error is usually returned by dbx.LoadFirst (that uses sqlx.Get, and then QueryRow).
// This error is NOT returned by lists loaded with dbx.LoadAll, sqlx.Select or db.Query, those return empty lists.
func IsErrNoRows(err error) bool {
	return err != nil && Is(err, sql.ErrNoRows)
}

// IsErrUniqueViolation checks if the error was returned by a DB exec that violated uniqueness.
// For example, when inserting a new row with the same unique id as another existing row.
func IsErrUniqueViolation(err error) bool {
	if err == nil {
		return false
	}
	err = errx.Unwrap(err)
	pqErr, ok := err.(*pq.Error)
	if !ok {
		return false
	}
	return pqErr.Code == "23505" // pg driver code for "unique_violation"
}

// NewErrUniqueConstraint can be used on tests or implementation stubs to simulate a "unique_violation" pq error.
func NewErrUniqueViolation(msg string) *pq.Error {
	return &pq.Error{
		Code:    pq.ErrorCode("23505"),
		Message: msg,
	}
}

// Is returns true if the error or any of the causes in the error chain is the target error.
func Is(err, target error) bool {
	if err == target {
		return true
	}
	if e, ok := err.(wrapper); ok {
		return Is(e.Cause(), target) // unwrap the error and try again recursively
	}
	return false
}

// Unwrap checks if the error has a Cause and returns that cause,
// otherwise returns the original error. It is not recursive (unlike pkg/errors.Cause(err)).
func Unwrap(err error) error {
	if err == nil {
		return nil
	}
	if e, ok := err.(wrapper); ok {
		return e.Cause()
	}
	return err
}

// StatusCode returns the HTTP status code (404, 500, etc) from the error chain.
// When the status is unknown returns 0, for example, plain errors like error.New("foobar") return status 0.
func StatusCode(err error) int {
	switch e := err.(type) {
	case *twitchclient.Error:
		return e.StatusCode
	case statusCoder:
		return e.StatusCode()
	case twirp.Error:
		return twirp.ServerHTTPStatusFromErrorCode(e.Code())
	case wrapper:
		return StatusCode(e.Cause()) // unwrap the error and try again
	default:
		return 0
	}
}

// IsInternalError checks if the error or any of the causes have a status other than 2xx or 3xx or 4xx.
// This can be used to check if an error returned by a client is due to service problems or not.
func IsInternalError(err error) bool {
	status := StatusCode(err)
	return status == 0 || status >= 500
}

// TwirpErrorFrom returns the equivalent twirp.Error from an HTTP client or database error.
// The returned error also responds to Cause and can be unwrapped by errx.Cause or pkg.errors.Cause.
// The code equivalence uses a best-guess mapping of known HTTP and database errors (see TwirpCodeFromStatus).
// This helper is useful to wrap trusted HTTP client errors as Twirp errors, similar to Visage's TrustedError helper.
func TwirpErrorFrom(err error) twirp.Error {
	switch e := err.(type) {
	case nil:
		return nil
	case twirp.Error:
		return e
	default: // probably an HTTP client error
		code := TwirpCodeFrom(err) // map client status to twirp code
		msg := Message(err)        // cleanup error message
		return &twirpErrWrapper{
			twerr: twirp.NewError(code, msg),
			cause: err,
		}
	}
}

// TwirpCodeFrom returns the twirp.Error code from the error or any of the causes in the error chain.
// If it is a twirp.Error, it returns the code.
// If it has an HTTP StatusCode, it makes the best guess to match an equivalent twirp.ErrorCode.
func TwirpCodeFrom(err error) twirp.ErrorCode {
	switch e := err.(type) {
	case nil:
		return twirp.NoError
	case twirp.Error:
		return e.Code()
	case *twitchclient.Error:
		return TwirpCodeFromStatus(e.StatusCode)
	case statusCoder:
		return TwirpCodeFromStatus(e.StatusCode())
	case wrapper:
		return TwirpCodeFrom(e.Cause()) // unwrap the error and try again
	default:
		return twirp.Internal
	}
}

// TwirpCodeFromStatus makes the best guess for the twirp.ErrorCode that could be used
// from common HTTP status codes. This can be useful to easily wrap HTTP client errors as Twirp errors.
func TwirpCodeFromStatus(statusCode int) twirp.ErrorCode {
	if statusCode >= 200 && statusCode < 400 {
		return twirp.NoError // 2xx and 3xx
	}

	switch statusCode {
	case 400, 422: // Bad Request, Unprocessable Entity (422 used by EMS sometimes on required fields)
		return twirp.InvalidArgument // could also be twirp.OutOfRange
	case 401: // Unauthorized
		return twirp.Unauthenticated
	case 403: // Forbidden
		return twirp.PermissionDenied // could also be twirp.ResourceExhausted
	case 404: // Not Found
		return twirp.NotFound // could also be twirp.BadRoute
	case 408: // Request Timeout
		return twirp.Canceled // could also be twirp.DeadlineExceeded
	case 409: // Conflict
		return twirp.AlreadyExists // could also be twirp.Aborted
	case 412: // Precondition Failed
		return twirp.FailedPrecondition
	case 0, 500: // Internal Server Error
		return twirp.Internal // could also be twirp.DataLoss, or twirp.Unknown
	case 501: // Not Implemented
		return twirp.Unimplemented
	case 503: // Service Unavailable
		return twirp.Unavailable
	default:
		return twirp.Unknown
	}
}

// Message reads the message in the error, skipping known prefixes like "Twirp error: ".
// Use this instead of just calling err.Error() for cleaner results.
func Message(err error) string {
	switch e := err.(type) {
	case twirp.Error:
		return e.Msg()
	case *twitchclient.Error:
		return e.Message
	case nil:
		return ""
	default:
		return err.Error()
	}
}

// ValidateUUID returns a twirp InvalidArgument error if the parameter is not a valid uuid.
func ValidateUUID(argKey, argVal string) error {
	if _, err := uuid.FromString(argVal); err != nil {
		return twirp.InvalidArgumentError(argKey, "invalid UUID format")
	}
	return nil
}

// ValidateRequiredArgs checks if a set of argument are not zero values.
// Returns error twirp.InvalidArgument with meta(arg.Key) on the first argument that is empty.
func ValidateRequiredArgs(args Args) twirp.Error {
	for _, arg := range args {
		if isZero(arg.Val) {
			return twirp.RequiredArgumentError(arg.Key)
		}
	}
	return nil
}

type Args []struct {
	Key string
	Val interface{}
}

// isZero returns true for zero-values like nil, "", 0, 0.0
func isZero(val interface{}) bool {
	return val == nil || val == reflect.Zero(reflect.TypeOf(val)).Interface()
}

type wrapper interface {
	Cause() error
}

type statusCoder interface {
	StatusCode() int
}

// twirpErrWrapper implements twirp.Error and wrapper (so the cause can be unwrapped)
type twirpErrWrapper struct {
	twerr twirp.Error
	cause error
}

func (e *twirpErrWrapper) Code() twirp.ErrorCode      { return e.twerr.Code() }
func (e *twirpErrWrapper) Msg() string                { return e.twerr.Msg() }
func (e *twirpErrWrapper) Meta(key string) string     { return e.twerr.Meta(key) }
func (e *twirpErrWrapper) MetaMap() map[string]string { return e.twerr.MetaMap() }
func (e *twirpErrWrapper) Error() string              { return e.twerr.Error() }
func (e *twirpErrWrapper) WithMeta(key string, val string) twirp.Error {
	return &twirpErrWrapper{twerr: e.twerr.WithMeta(key, val), cause: e.cause}
}
func (e *twirpErrWrapper) Cause() error { return e.cause }
