package jwt

import (
	"bytes"
	"encoding/base64"
	"encoding/json"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/common/jwt"
)

type Claims struct {
	Audience string `json:"aud"`
	Issuer   string `json:"iss"`
	Subject  string `json:"sub"`
	Expires  int64  `json:"exp"` // unix timestamp
}

// EncodeRS256 base64-encodes an RS256 JWT, signed with your private key.
func EncodeRS256(privateKey []byte, claims Claims) ([]byte, error) {
	key, err := jwt.ParseRSAPrivateKey(privateKey)
	if err != nil {
		return nil, errx.Wrap(err, "jwt: failed to parse rsa private key")
	}

	rs256 := jwt.RS256(key)

	encoded, err := jwt.Encode(jwt.NewHeader(rs256), claims, rs256)
	if err != nil {
		return nil, errx.Wrap(err, "jwt: failed to encode")
	}

	return encoded, nil
}

// DecodeRS256 base64-decodes an RS256 JWT and verifies the signature with the public key.
func DecodeRS256(publicKey []byte, encoded []byte) (Claims, error) {
	key, err := jwt.ParseRSAPublicKey(publicKey)
	if err != nil {
		return Claims{}, errx.Wrap(err, "jwt: failed to parse rsa public key")
	}

	fragment, err := jwt.Parse(encoded)
	if err != nil {
		return Claims{}, errx.Wrap(err, "jwt: failed to parse serialized jwt")
	}

	if err = fragment.Validate(jwt.RSAValidateOnly(jwt.RS256, key)); err != nil {
		return Claims{}, errx.Wrap(err, "jwt: invalid rs256 jwt")
	}

	decoder := json.NewDecoder(base64.NewDecoder(base64.URLEncoding, bytes.NewReader(fragment.Claims())))

	var claims Claims
	if err = decoder.Decode(&claims); err != nil {
		return Claims{}, errx.Wrap(err, "jwt: failed to decode serialized jwt claims")
	}

	return claims, nil
}
