package middleware

import (
	"context"
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"github.com/twitchtv/twirp/ctxsetters"
)

func TestStatsdServerHooks_Success(t *testing.T) {
	statter := &fakeStatter{}
	hooks := NewStatsdServerHooks(statter)

	ctx := context.Background()
	ctx, _ = hooks.RequestReceived(ctx)

	ctx = ctxsetters.WithMethodName(ctx, "FooMethod")
	ctx = ctxsetters.WithServiceName(ctx, "FooService")
	ctx = ctxsetters.WithStatusCode(ctx, 200)
	hooks.ResponseSent(ctx)

	require.Equal(t, "twirpservice.FooService.FooMethod.200", statter.incMetric)
	require.Equal(t, "twirpservice.FooService.FooMethod.200", statter.timingDurationMetric)
}

func TestStatsdServerHooks_InvalidContext(t *testing.T) {
	statter := &fakeStatter{}
	hooks := NewStatsdServerHooks(statter)

	ctx := context.Background()
	ctx, _ = hooks.RequestReceived(ctx)
	hooks.ResponseSent(ctx)

	require.Equal(t, "twirpservice.hooksfail.hooksfail.0", statter.incMetric)
	require.Equal(t, "twirpservice.hooksfail.hooksfail.0", statter.timingDurationMetric)
}

func TestStatsdServerHooks_InvalidRequestReceived(t *testing.T) {
	statter := &fakeStatter{}
	hooks := NewStatsdServerHooks(statter)

	ctx := context.Background()
	// ctx, _ = hooks.RequestReceived(ctx) // simulate fatal twirp issue, hook not called for some reason
	ctx = ctxsetters.WithMethodName(ctx, "FooMethod")
	ctx = ctxsetters.WithStatusCode(ctx, 500)
	hooks.ResponseSent(ctx)

	require.Equal(t, "twirpservice.hooksfail.FooMethod_nostart.500", statter.incMetric)
	require.Equal(t, "twirpservice.hooksfail.FooMethod_nostart.500", statter.timingDurationMetric)
}

//
// Test Helpers
//

type fakeStatter struct {
	incMetric            string
	timingDurationMetric string
}

func (s *fakeStatter) Inc(metric string, val int64, rate float32) error {
	s.incMetric = metric
	return nil
}

func (s *fakeStatter) TimingDuration(metric string, val time.Duration, rate float32) error {
	s.timingDurationMetric = metric
	return nil
}
