package utils

import (
	"fmt"
	"reflect"
	"strconv"

	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/aws/ec2metadata"
	"github.com/aws/aws-sdk-go/aws/session"
)

// ErrCodeEC2MetadataRequestError for service response error code
// "EC2MetadataRequestError".
//
// Cannot connect to host, likely not running in EC2
const ErrCodeEC2MetadataRequestError = "EC2MetadataRequestError"

// IfThenElse evaluates a condition, if true returns the first parameter otherwise the second
func IfThenElse(condition bool, tru interface{}, flse interface{}) interface{} {
	if condition {
		return tru
	}
	return flse
}

// IfThen evaluates a condition, if true returns the first parameter otherwise nil
func IfThen(condition bool, tru interface{}) interface{} {
	if condition {
		return tru
	}
	return nil
}

func IfSliceLength(slice interface{}, index int) interface{} {
	s := reflect.ValueOf(slice)
	if s.Kind() != reflect.Slice {
		panic("IfSliceLength cannot be passed a non-slice type (" + s.Kind().String() + ")")
	}

	if s.Len() > index {
		return s.Index(index).Interface()
	}

	return nil
}

func IfStringSliceLength(slice []string, index int) string {

	if len(slice) > index {
		return slice[index]
	}

	return ""
}

func ParseUnknownToString(item interface{}) (string, error) {
	switch item := item.(type) {

	case string:
		return item, nil

	case float32:
		return strconv.FormatFloat(float64(item), 'f', -1, 32), nil
	case float64:
		return strconv.FormatFloat(item, 'f', -1, 64), nil

	case int, int64:
		return fmt.Sprintf("%d", item), nil

	}

	return "", fmt.Errorf("Unsupported Type to string (%T)", item)
}

func GetCurrentRegion() (string, error) {
	// This is much easier with GoSDK v2, but here is how we have to do it in v1...
	metaSession, err := session.NewSession()
	if err != nil {
		return "", err
	}

	metaClient := ec2metadata.New(metaSession)
	region, err := metaClient.Region()

	if err != nil {
		if aerr, ok := err.(awserr.Error); ok {
			switch aerr.Code() {
			case ErrCodeEC2MetadataRequestError:
				return "us-west-2", nil
			}
		}
		return "", err
	}

	return region, nil
}
