package extensionreviewsserver

import (
	"context"
	"time"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/twitchtv/twirp"
)

func (s *Server) SetStateApproved(ctx context.Context, params *rbacrpc.SetStateReviewerRequest) (*rbacrpc.Empty, error) {
	err := s.setStateByReviewer(ctx, "approved", params)
	return &rbacrpc.Empty{}, err
}

func (s *Server) setStateByReviewer(ctx context.Context, state string, params *rbacrpc.SetStateReviewerRequest) error {
	if err := validateSetStateReviewerRequest(params); err != nil {
		return err
	}

	// Make sure that the Authorization header is present and is valid.
	oauthToken := auth.GetRawAuthorizationToken(ctx)
	if oauthToken == "" {
		return twirp.NewError(twirp.Unauthenticated, "Mising Authorization header with OAuth token")
	}
	reviewerTwitchID := auth.GetTwitchID(ctx)
	if reviewerTwitchID == "" {
		return twirp.NewError(twirp.Unauthenticated, "Could not identify reviewer")
	}

	// Get Cartman token
	// Need multiple capabilities to make sure reviewers can perform the action:
	//  * "extensions::view_all_extensions": access the extension from another user
	//  * "extensions::edit_version": requires the extension_id, allows to modify the manifest.
	//  * "extensions::review_extensions": globally protects "approved" and "pending_action" states, and only those.
	capabilities := "extensions::view_all_extensions,extensions::edit_version,extensions::review_extensions"
	authFnParams := map[string]string{"extension_id": params.ExtensionId}
	cartmanToken, err := s.Cartman.GetToken(ctx, oauthToken, capabilities, authFnParams)
	if err != nil {
		return errorutil.TwirpErrorFrom(err)
	}

	// Transition to state in EMS backend
	err = s.Extensions.TransitionExtensionState(ctx, params.ExtensionId, params.ExtensionVersion, state, cartmanToken)
	if err != nil {
		return errorutil.TwirpErrorFrom(err)
	}

	s.registerExtensionReviewLog(ctx, state, reviewerTwitchID, params)

	return nil
}

func (s *Server) registerExtensionReviewLog(ctx context.Context, state, reviewerTwitchID string, params *rbacrpc.SetStateReviewerRequest) {
	now := time.Now()
	reviewLog := &rbacrpc.ExtensionReviewLog{
		CreatedAt:         now.Format(time.RFC3339),
		State:             state,
		ExtensionId:       params.ExtensionId,
		ExtensionVersion:  params.ExtensionVersion,
		ReviewerTwitchId:  reviewerTwitchID,
		ReviewReasonCodes: params.ReviewReasonCodes,
		ReviewReason:      params.ReviewReason,
		SalesforceCaseId:  "",
	}

	// Take SalesforceCaseId from previous review record
	prevReviewLog, err := s.DBExtensionReviewLogs.GetLastInReview(ctx, params.ExtensionId, params.ExtensionVersion)
	if !errorutil.IsErrNoRows(err) {
		if err != nil {
			logErrorRegisteringReviewLog(ctx, err, reviewLog)
		} else {
			reviewLog.SalesforceCaseId = prevReviewLog.SalesforceCaseId
		}
	}

	err = s.DBExtensionReviewLogs.Insert(ctx, reviewLog)
	if err != nil {
		logErrorRegisteringReviewLog(ctx, err, reviewLog)
	}
}

func logErrorRegisteringReviewLog(ctx context.Context, err error, r *rbacrpc.ExtensionReviewLog) {
	// Don't fail the request, because the extension is already approved and the logs are not critical.
	// However, make sure we report the error and get paged by this because something is very wrong with the DB.
	fields := logx.Fields{ // lost record fields so we can be re-insert form the logs
		"CreatedAt":         r.CreatedAt,
		"State":             r.State,
		"ExtensionId":       r.ExtensionId,
		"ExtensionVersion":  r.ExtensionVersion,
		"ReviewerTwitchId":  r.ReviewerTwitchId,
		"ReviewReasonCodes": r.ReviewReasonCodes,
		"ReviewReason":      r.ReviewReason,
		"SalesforceCaseId":  r.SalesforceCaseId,
	}
	logx.Error(ctx, errx.Wrap(err, "Failed to insert ExtensionReviewLog"), fields)
}

func validateSetStateReviewerRequest(params *rbacrpc.SetStateReviewerRequest) error {
	if params.ExtensionId == "" {
		return twirp.RequiredArgumentError("extension_id")
	}
	if params.ExtensionVersion == "" {
		return twirp.RequiredArgumentError("extension_version")
	}
	return nil
}
