package privacyserver

import (
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/backend/actionhistories"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	rcommon "code.justin.tv/devrel/devsite-rbac/common"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpcserver"
	"context"
	"errors"
	"fmt"
)

func (s *Server) Start(ctx context.Context, _ *rbacrpc.Empty) (*rbacrpc.Empty, error) {
	go s.PrivacyJob(ctx)
	return &rbacrpc.Empty{}, nil
}

func (s *Server) StartAdmin(ctx context.Context, id *rbacrpc.Id) (*rbacrpc.Empty, error) {
	if !auth.IsWhitelistAdmin(ctx) {
		return &rbacrpc.Empty{}, nil
	}

	err := s.RunIndividualUser(ctx, id.Id)
	return &rbacrpc.Empty{}, err
}

func (s *Server) AwsCloudwatch(ctx context.Context, request *rbacrpc.AwsCloudwatchRequest) (*rbacrpc.Empty, error) {

	if request.ApiKey == "" {
		return &rbacrpc.Empty{}, fmt.Errorf("Malformed ApiKey - %v", request.ApiKey)
	}

	if request.ApiKey != s.ApiKey {
		return &rbacrpc.Empty{}, errors.New("invalid_access_key")
	}

	go s.PrivacyJob(context.Background())
	return &rbacrpc.Empty{}, nil
}

func (s *Server) PrivacyJob(ctx context.Context) {
	pendingIds, errPdms := s.PDMS.GetOutstandingDeletionRequests(ctx)
	if errPdms != nil {
		logx.Error(ctx, errPdms, logx.Fields{
			"message": "Failed to get deletion requests",
		})

		return
	}

	for _, userId := range pendingIds {
		_, err := s.ProcessDeletionRequest(ctx, userId)

		if err != nil {
			logx.Error(ctx, err, logx.Fields{
				"message": fmt.Sprintf("Failed to run deletion for %v", userId),
			})
			continue
		}

		err = s.PDMS.MarkDeleted(ctx, userId)
		if err != nil {
			logx.Error(ctx, err, logx.Fields{
				"message": fmt.Sprintf("Failed to mark deleted for %v", userId),
			})
			continue
		}

		logx.Info(ctx, fmt.Sprintf("Successfully ran privacy job on %v.", userId))
	}

	logx.Info(ctx, fmt.Sprintf("Successfully ran privacy job on %v users.", len(pendingIds)))
}

func (s *Server) RunIndividualUser(ctx context.Context, twitchID string) error {
	_, err := s.ProcessDeletionRequest(ctx, twitchID)

	if err != nil {
		logx.Error(ctx, err, logx.Fields{
			"message": fmt.Sprintf("Failed to run individual privacy user (%s)", twitchID),
		})
	} else {
		logx.Info(ctx, fmt.Sprintf("IndividualUser Success on (%s)", twitchID))
	}

	return err
}

// Takes in a Twitch ID and runs the deletion process in order.
func (s *Server) ProcessDeletionRequest(ctx context.Context, userTwitchID string) (string, error) {
	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	companies, err := s.Memberships.GetMembershipsByAccount(ctx, userTwitchID)

	if err != nil {
		return "", err
	}

	for _, companyMembership := range companies {
		if companyMembership.Role == rcommon.OwnerRole {
			err := s.MigrateMemberCompany(ctx, userTwitchID, companyMembership.CompanyID)

			if err != nil {
				return "", err
			}
		}

		dErr := s.Memberships.DeleteMembership(ctx, companyMembership.CompanyID, userTwitchID)

		// This can be no rows because if the entire company is deleted all members get deleted with it
		if dErr != nil && !errorutil.IsErrNoRows(dErr) {
			return "", dErr
		}

	}

	uwErr := s.ViennaUserWhitelist.RemoveUserFromWhitelist(ctx, &rbacrpc.RemoveUserFromWhitelistRequest{
		TwitchId: userTwitchID,
	})

	if uwErr != nil && !errorutil.IsErrNoRows(uwErr) {
		return "", uwErr
	}

	ahErr := s.ActionHistories.StripUserActionHistory(ctx, userTwitchID)

	if ahErr != nil {
		return "", ahErr
	}

	daErr := s.DeveloperApplications.DeleteDeveloperApplicationByAccount(ctx, userTwitchID)

	if daErr != nil {
		return "", daErr
	}

	caErr := s.Backend.DeleteCompanyApplicationsByAccount(ctx, userTwitchID)

	if caErr != nil {
		return "", caErr
	}

	ciErr := s.Backend.DeleteAllCompanyInvitesByTwitchId(ctx, userTwitchID)

	if ciErr != nil {
		return "", ciErr
	}

	return "", s.Backend.Commit(ctx)
}

func (s *Server) MigrateMemberCompany(ctx context.Context, userTwitchID string, companyID string) error {
	allAdmins, err := s.Memberships.GetAdminsByCompany(ctx, companyID)

	if err != nil {
		return err
	}

	if len(allAdmins) == 0 {
		return s.DeleteCompany(ctx, companyID)
	} else {
		err = s.Memberships.MigrateOwner(ctx, companyID, userTwitchID, allAdmins[0].TwitchID)

		if err == nil {
			s.auditUserRoleChange(ctx, rbacrpcserver.UserRoleAction{
				EntityTwitchID: allAdmins[0].TwitchID,
				ActionFormat:   "Edit: Assigned company role of %s due to old owner account deletion",
				Role:           "Owner",
				CompanyID:      companyID,
			})
		}

		return err
	}
}

func (s *Server) DeleteCompany(ctx context.Context, companyID string) error {

	err := s.Backend.DeleteGameApplicationsByCompany(ctx, companyID)

	if err != nil {
		return err
	}

	err = s.DeveloperApplications.DeleteDeveloperApplicationByCompany(ctx, companyID)

	if err != nil {
		return err
	}

	err = s.Backend.DeleteResourcesByCompany(ctx, companyID)

	if err != nil {
		return err
	}

	err = s.Memberships.DeleteCompanyMemberships(ctx, companyID)

	if err != nil {
		return err
	}

	return s.Backend.DeleteCompany(ctx, &rbacrpc.DeleteCompanyRequest{
		Id: companyID,
	})
}

func (s *Server) auditUserRoleChange(ctx context.Context, a rbacrpcserver.UserRoleAction) {
	actingId := auth.GetTwitchID(ctx)
	if actingId == "" {
		actingId = "-1"
	}

	s.ActionHistories.InsertActionHistory(ctx, &actionhistories.ActionHistory{
		UserTwitchID: actingId,
		Action:       fmt.Sprintf(a.ActionFormat, a.Role),
		EntityType:   "UserRole",
		EntityID:     a.EntityTwitchID,
		CompanyID:    common.NewSQLNullString(a.CompanyID),
	})
}
