package rbacadminserver

import (
	"context"

	"github.com/twitchtv/twirp"

	"code.justin.tv/devrel/devsite-rbac/backend/featuregating"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

func (s *Server) GetFeatureGatingValueInBool(ctx context.Context, params *rbacrpc.GetFeatureGatingValueRequest) (*rbacrpc.GetFeatureGatingValueResponse, error) {
	if err := s.ValidateWhitelistAdmin(ctx); err != nil {
		return nil, err
	}
	value, err := s.FeatureGating.BoolFeatureGate(ctx, params.Key)
	if err != nil {
		return nil, err
	}
	return &rbacrpc.GetFeatureGatingValueResponse{
		Key:         params.Key,
		ValueInBool: value,
	}, nil
}

func (s *Server) SetFeatureGatingBoolValue(ctx context.Context, params *rbacrpc.SetFeatureGatingBoolValueRequest) (*rbacrpc.Empty, error) {
	if err := s.ValidateWhitelistAdmin(ctx); err != nil {
		return nil, err
	}
	err := s.FeatureGating.SetBoolFeatureGate(ctx, params.Key, params.ValueInBool)
	return &rbacrpc.Empty{}, err
}

func (s *Server) GetFeatureGatingValueInStrings(ctx context.Context, params *rbacrpc.GetFeatureGatingValueRequest) (*rbacrpc.GetFeatureGatingValueResponse, error) {
	if err := s.ValidateWhitelistAdmin(ctx); err != nil {
		return nil, err
	}
	result, err := s.FeatureGating.StringsFeatureGate(ctx, params.Key)
	if err != nil {
		return nil, err
	}
	return &rbacrpc.GetFeatureGatingValueResponse{
		Key:            params.Key,
		ValuesInString: result,
	}, nil
}

func (s *Server) UpdateFeatureGatingValue(ctx context.Context, params *rbacrpc.UpdateFeatureGatingValueRequest) (*rbacrpc.GetFeatureGatingValueResponse, error) {
	if err := s.ValidateWhitelistAdmin(ctx); err != nil {
		return nil, err
	}
	result, err := s.FeatureGating.UpdateStringsFeatureGate(ctx, params.Key, params.ValueInString, params.Action)
	if err != nil {
		switch errorutil.Unwrap(err) {
		case featuregating.ErrValueAlreadyExists:
			return nil, twirp.NewError(twirp.InvalidArgument, "value already exists")
		case featuregating.ErrValueDoesntExist:
			return nil, twirp.NewError(twirp.InvalidArgument, "value doesn't exist")
		}
		return nil, err
	}
	return &rbacrpc.GetFeatureGatingValueResponse{
		Key:            params.Key,
		ValuesInString: result,
	}, nil
}
