package rbacadminserver

import (
	"context"
	"fmt"
	"strings"

	"code.justin.tv/devrel/devsite-rbac/backend/common"

	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"

	"code.justin.tv/devrel/devsite-rbac/backend/actionhistories"
	"code.justin.tv/devrel/devsite-rbac/backend/companyresources"

	"github.com/twitchtv/twirp"

	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

func (s *Server) OffboardResource(ctx context.Context, params *rbacrpc.OffboardResourceRequest) (empty *rbacrpc.Empty, err error) {
	err = s.ValidateWhitelistAdmin(ctx)
	if err != nil {
		return nil, err
	}

	company, err := s.validateOffboardResource(ctx, params)
	if err != nil {
		return nil, err
	}

	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	resource := &companyresources.ResourceType{
		CompanyID:  params.CompanyId,
		ExternalId: params.ResourceId,
		Type:       params.ResourceType,
	}

	// Delete resource
	if err := s.Backend.DeleteResource(ctx, resource); err != nil {
		return nil, err
	}

	if err := s.Backend.Commit(ctx); err != nil {
		return nil, err
	}

	s.auditOffboardResource(ctx, resource, company)

	return &rbacrpc.Empty{}, err
}

func (s *Server) validateOffboardResource(ctx context.Context, params *rbacrpc.OffboardResourceRequest) (company *rbacrpc.Company, err error) {
	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"company_id", params.CompanyId},
		{"resource_id", params.ResourceId},
		{"resource_type", params.ResourceType},
	}); err != nil {
		return nil, err
	}

	company, err = s.Backend.SelectCompany(ctx, params.CompanyId)
	if err != nil {
		err = twirp.NotFoundError("company not found")
		return
	}

	resourceTypes, _, err := s.Backend.FindResources(ctx, params.CompanyId, params.ResourceType, params.ResourceId, 1, 0)
	if err != nil {
		return nil, err
	}

	if len(resourceTypes) == 0 {
		return nil, twirp.NotFoundError("company does not own resource")
	}

	return
}

func (s *Server) auditOffboardResource(ctx context.Context, resource *companyresources.ResourceType, company *rbacrpc.Company) {
	titleResourceType := strings.Title(resource.Type)
	s.ActionHistories.InsertActionHistory(ctx, &actionhistories.ActionHistory{
		UserTwitchID: auth.GetTwitchID(ctx),
		Action:       fmt.Sprintf("%s removed", titleResourceType),
		EntityType:   titleResourceType,
		EntityID:     resource.ExternalId,
		CompanyID:    common.NewSQLNullString(company.Id),
	})
}
