package rbacadminserver

import (
	"context"
	"fmt"
	"math/rand"
	"sync"
	"time"

	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/clients/passport"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/utils"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"

	multierror "github.com/hashicorp/go-multierror"
	"github.com/pkg/errors"
	"github.com/sirupsen/logrus"

	"github.com/twitchtv/twirp"
)

const (
	shadowAccountRole          = "Shadow_Account"
	companyMemberLimit         = 1000
	usernameRandomStringLength = 16
	randomStringCharSet        = "abcdefghijklmnopqrstuvwxyz" + "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"
	defaultRetryTimes          = 5
)

// CreateShadowAccount takes in a twitch ID and a companyID, it does:
// 1. call internal passport API to sign up a new account and call user service to get the twitch ID.
// 2. call user service to modify 2 fields of the channel to make it hidden from search and discovery
// 3. call Chat’s service to update the whitelist of the shadow account channel with the org’s members.
// 4. call Nioh to update the RestrictionType and ExemptionType for the shadow account channel.
// 5. Update the table “membership” with a new record with the role “ShadowAccounts”.
func (s *Server) CreateShadowAccount(ctx context.Context, params *rbacrpc.CreateShadowAccountRequest) (*rbacrpc.CreateShadowAccountResponse, error) {
	err := s.ValidateWhitelistAdmin(ctx)
	if err != nil {
		return nil, err
	}

	err = s.validateCreateShadowAccountParams(ctx, params)
	if err != nil {
		return nil, err
	}

	mutex := &sync.Mutex{}
	errs := make(map[string]error, len(params.GetUserInput()))
	userIDMap := make(map[string]string, len(params.GetUserInput()))
	usernameMap := make(map[string]string, len(params.GetUserInput()))
	records := make([]*rbacrpc.CreateShadowAccountRecord, len(params.GetUserInput()))

	wg := &sync.WaitGroup{}
	var clueErr error
	for _, input := range params.UserInput {
		if input.Email == "" {
			continue
		}
		wg.Add(1)
		go func(firstName, lastName, email, clientID, companyID string) {
			defer wg.Done()
			channelID, username, err := s.signupNewAccount(ctx, email, clientID)
			if err != nil {
				mutex.Lock()
				errs[email] = err
				mutex.Unlock()
				return
			}
			logrus.Debugf("Sign up user using email: %s, channelID: %s\n", email, channelID)
			mutex.Lock()
			userIDMap[email] = channelID
			usernameMap[email] = username
			mutex.Unlock()
			err = s.updateRelatedServices(ctx, channelID, companyID, firstName, lastName, email)
			if err != nil {
				mutex.Lock()
				errs[email] = err
				mutex.Unlock()
			}
		}(input.FirstName, input.LastName, input.Email, params.ClientId, params.CompanyId)
	}
	wg.Wait()
	clueErr = s.updateChannelAllowedChatters(ctx, params.CompanyId)

	var multierr *multierror.Error
	for email, err := range errs {
		multierr = multierror.Append(multierr, errors.Wrap(err, "failed to create shadow account using email: "+email))
	}

	if clueErr != nil {
		logrus.Debugf("Error when updating channels allowed users for company %s: %s\n", params.CompanyId, clueErr.Error())
		multierr = multierror.Append(multierr, clueErr)
	}

	for idx, input := range params.UserInput {
		if input.Email == "" {
			continue
		}
		nullableUserID := userIDMap[input.Email]
		nullableUsername := usernameMap[input.Email]
		nullableErr := errs[input.Email]
		var errorMessage = ""
		if nullableErr != nil {
			errorMessage = nullableErr.Error()
		}
		records[idx] = &rbacrpc.CreateShadowAccountRecord{
			Email:        input.Email,
			TwitchId:     nullableUserID,
			Username:     nullableUsername,
			ErrorMessage: errorMessage,
		}
	}

	var errMsg = ""
	if multierr.ErrorOrNil() != nil {
		errMsg = multierr.Error()
	}
	// don't return error so that the front end will get the error details
	return &rbacrpc.CreateShadowAccountResponse{
		Error:   errMsg,
		Records: records,
	}, nil
}

// signupNewAccount returns
// twitch ID (empty string if it was created unsuccessfully)
// username (empty string if it was created unsuccessfully)
// error if there is any
func (s *Server) signupNewAccount(ctx context.Context, email string, clientID string) (string, string, error) {
	if email == "" {
		return "", "", twirp.NewError(twirp.InvalidArgument, "email is missing.")
	}

	username := generateShadowAccountUsername()
	_, err := s.Passport.SignUpNewAccount(ctx, passport.CreateNewAccountRequest{
		ClientID: clientID,
		Email:    email,
		Username: username,
		// just use some fake data for the birth day since it's not important
		Birthday: passport.Birthday{
			Day:   1,
			Month: 1,
			Year:  2000,
		},
	})
	if err != nil {
		return "", "", err
	}

	userProperties, err := s.Users.GetUserByLogin(ctx, username)
	if err != nil {
		return "", "", err
	}
	return userProperties.ID, username, nil
}

func generateShadowAccountUsername() string {
	seededRand := rand.New(rand.NewSource(time.Now().UnixNano()))
	b := make([]byte, usernameRandomStringLength)
	for i := range b {
		b[i] = randomStringCharSet[seededRand.Intn(len(randomStringCharSet))]
	}
	return "shadow_" + string(b)
}

func (s *Server) updateRelatedServices(ctx context.Context, channelID, companyID, firstName, lastName, email string) error {
	var appliedFirstName = firstName
	if appliedFirstName == "" {
		appliedFirstName = "shadow_" + channelID[len(channelID)-4:]
	}

	var appliedLastName = lastName
	if appliedLastName == "" {
		appliedLastName = "shadow"
	}

	err := s.updateShadowAccountMembership(ctx, channelID, companyID, appliedFirstName, appliedLastName, email)
	if err != nil {
		return err
	}

	callerID := auth.GetTwitchID(ctx)
	wg := &sync.WaitGroup{}
	var userServiceErr error
	var niohErr error

	wg.Add(2)
	go func() {
		defer wg.Done()
		userServiceErr = utils.WithRetry(func() error {
			return s.Channels.HideChannel(ctx, channelID)
		}, defaultRetryTimes)
	}()

	go func() {
		defer wg.Done()
		niohErr = utils.WithRetry(func() error {
			return s.Nioh.SetOrgRestrictionOnChannel(ctx, callerID, channelID)
		}, defaultRetryTimes)
	}()
	wg.Wait()

	var multierr *multierror.Error
	if userServiceErr != nil {
		logrus.Debugf("Error when updating channel %s properties: %s\n", channelID, userServiceErr.Error())
		multierr = multierror.Append(multierr, errors.Wrap(userServiceErr, "failed to hdie channel in user service"))
	}

	if niohErr != nil {
		logrus.Debugf("Error when updating nioh restrictions for channel %s: %s\n", channelID, niohErr.Error())
		multierr = multierror.Append(multierr, errors.Wrap(niohErr, "failed to set restriction"))
	}

	if multierr != nil {
		err = s.Memberships.DeleteMembership(ctx, companyID, channelID)
		if err != nil {
			multierr = multierror.Append(multierr, errors.Wrap(err, "failed to remove invalid membership"))
		}

		return multierr.ErrorOrNil()
	}
	return nil
}

func (s *Server) updateShadowAccountMembership(ctx context.Context, twitchID string, companyID string, firstName string, lastName string, devEmail string) error {
	newMembership := &memberships.Membership{
		CompanyID: companyID,
		TwitchID:  twitchID,
		Role:      shadowAccountRole,
		FirstName: firstName,
		LastName:  lastName,
		DevTitle:  shadowAccountRole,
		DevEmail:  devEmail,
	}
	return s.Memberships.InsertMembership(ctx, newMembership)
}

func (s *Server) validateCreateShadowAccountParams(ctx context.Context, params *rbacrpc.CreateShadowAccountRequest) error {
	if params.CompanyId == "" {
		return twirp.NewError(twirp.InvalidArgument, "company_id is missing.")
	}

	if params.ClientId == "" {
		return twirp.NewError(twirp.InvalidArgument, "client_id is missing.")
	}

	if len(params.UserInput) == 0 {
		return twirp.NewError(twirp.InvalidArgument, "user_input is missing")
	}

	return nil
}

func (s *Server) updateChannelAllowedChatters(ctx context.Context, companyID string) error {
	members, _, err := s.Memberships.ListMemberships(ctx, memberships.ListMembershipsParams{
		CompanyID: companyID,
		Limit:     companyMemberLimit,
	})
	if err != nil {
		return err
	}

	if len(members) == 0 {
		return fmt.Errorf("companyID %s might be invalid, found 0 members", companyID)
	}

	var shadowAccountIDs []string
	var memberIDs = make([]string, len(members))
	for idx, member := range members {
		memberIDs[idx] = member.TwitchID
		if member.Role == shadowAccountRole {
			shadowAccountIDs = append(shadowAccountIDs, member.TwitchID)
		}
	}

	return s.Clue.UpdateAllowedChattersForChannels(ctx, memberIDs, shadowAccountIDs)
}
