package rbacadminserver

import (
	"context"
	"testing"

	"code.justin.tv/devrel/devsite-rbac/clients/passport"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/require"
)

const (
	companyID = "test_company_id"
	clientID  = "test_client_id"
)

func mockAdminContext() context.Context {
	return context.WithValue(context.Background(), auth.ViennaUserWhitelistRoleKey, rbacrpc.WhitelistUserRole_ADMIN)
}

func TestCreateShadowAccount_Unauthorized(t *testing.T) {
	testServer := NewTestAdminServer()
	badRequest := rbacrpc.CreateShadowAccountRequest{
		CompanyId: "company_id",
		ClientId:  "client_id",
		UserInput: []*rbacrpc.CreateShadowAccountUserInput{},
	}

	_, err := testServer.Server.CreateShadowAccount(context.Background(), &badRequest)

	require.EqualError(t, err, "twirp error permission_denied: User is unauthorized to take the action")
}

func TestCreateShadowAccount_BadRequest(t *testing.T) {
	testServer := NewTestAdminServer()
	badRequest := rbacrpc.CreateShadowAccountRequest{
		CompanyId: "company_id",
		ClientId:  "client_id",
		UserInput: []*rbacrpc.CreateShadowAccountUserInput{},
	}

	_, err := testServer.Server.CreateShadowAccount(mockAdminContext(), &badRequest)

	require.EqualError(t, err, "twirp error invalid_argument: user_input is missing")
}

func TestCreateShadowAccount(t *testing.T) {
	testServer := NewTestAdminServer()
	mockEmails := []string{"testuser1@test.com", "testuser2@test.com", "testuser3@test.com"}
	mockUserIDs := []string{"1245114", "3616113", "2164146"}
	request := rbacrpc.CreateShadowAccountRequest{
		CompanyId: companyID,
		ClientId:  clientID,
		UserInput: []*rbacrpc.CreateShadowAccountUserInput{
			&rbacrpc.CreateShadowAccountUserInput{
				Email: mockEmails[0],
			},
			&rbacrpc.CreateShadowAccountUserInput{
				Email: mockEmails[1],
			},
			&rbacrpc.CreateShadowAccountUserInput{
				Email: mockEmails[2],
			},
		},
	}

	testServer.MockPassport.SignUpNewAccountReturns(passport.CreateNewAccountResponse{
		AccessToken: "access token",
	}, nil)
	testServer.MockUsers.GetUserByLoginReturnsOnCall(0, &models.Properties{ID: mockUserIDs[0]}, nil)
	testServer.MockUsers.GetUserByLoginReturnsOnCall(1, &models.Properties{ID: mockUserIDs[1]}, nil)
	testServer.MockUsers.GetUserByLoginReturnsOnCall(2, &models.Properties{ID: mockUserIDs[2]}, nil)
	testServer.MockMemberships.InsertMembershipReturns(nil)
	testServer.MockChannels.HideChannelReturns(nil)
	testServer.MockNioh.SetOrgRestrictionOnChannelReturns(nil)
	testServer.MockClue.UpdateAllowedChattersForChannelsReturns(nil)

	resp, err := testServer.Server.CreateShadowAccount(mockAdminContext(), &request)
	require.NoError(t, err)
	require.Equal(t, 3, len(resp.Records))
	require.True(t, equalSlice(mockUserIDs, []string{resp.Records[0].TwitchId, resp.Records[1].TwitchId, resp.Records[2].TwitchId}))
	require.True(t, equalSlice(mockEmails, []string{resp.Records[0].Email, resp.Records[1].Email, resp.Records[2].Email}))
	require.NotEmpty(t, resp.Records[0].Username)
	require.NotEmpty(t, resp.Records[1].Username)
	require.NotEmpty(t, resp.Records[2].Username)
	require.Empty(t, resp.Records[0].ErrorMessage)
	require.Empty(t, resp.Records[1].ErrorMessage)
	require.Empty(t, resp.Records[2].ErrorMessage)
}

func equalSlice(strs1 []string, strs2 []string) bool {
	if len(strs1) != len(strs2) {
		return false
	}

	m := make(map[string]struct{}, len(strs1))

	for _, str := range strs1 {
		m[str] = struct{}{}
	}

	for _, str := range strs2 {
		if _, ok := m[str]; !ok {
			return false
		}
	}

	return true
}
