package rbacadminserver

import (
	"context"
	"fmt"

	"code.justin.tv/devrel/devsite-rbac/backend/common"

	"code.justin.tv/devrel/devsite-rbac/backend/actionhistories"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpcserver"

	"github.com/twitchtv/twirp"
)

func (s *Server) TransferCompanyOwnership(ctx context.Context, params *rbacrpc.TransferCompanyOwnershipRequest) (*rbacrpc.Empty, error) {
	err := s.ValidateWhitelistAdmin(ctx)
	if err != nil {
		return nil, err
	}

	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"company_id", params.CompanyId},
		{"twitch_id_to_promote", params.TwitchIdToPromote},
	}); err != nil {
		return nil, err
	}

	if err := errorutil.ValidateUUID("company_id", params.CompanyId); err != nil {
		return nil, err
	}

	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	promoteMemb, err := s.Memberships.GetMembership(ctx, params.CompanyId, params.TwitchIdToPromote)
	if errorutil.IsErrNoRows(err) {
		return nil, twirp.NewError(twirp.FailedPrecondition, "twitch_id_to_promote must be in company")
	} else if err != nil {
		return nil, err
	}

	ownerTwitchID := ""
	ownerList, total, err := s.Memberships.ListMemberships(ctx, memberships.ListMembershipsParams{
		CompanyID: params.CompanyId,
		Role:      "Owner",
		Limit:     1,
	})
	if err != nil {
		return nil, err
	}
	if total > 0 {
		// Update owner
		formattedRole, err := rbacrpcserver.SanitizeRole("owners_new_role", params.OwnersNewRole)
		if err != nil {
			return nil, err
		}
		params.OwnersNewRole = formattedRole

		owner := ownerList[0]
		ownerTwitchID = owner.TwitchID
		owner.Role = params.OwnersNewRole

		if err := s.Memberships.UpdateMembership(ctx, &owner); err != nil {
			return nil, err
		}
	}

	promoteMemb.Role = "Owner"
	if err := s.Memberships.UpdateMembership(ctx, &promoteMemb); err != nil {
		return nil, err
	}

	if err := s.Backend.Commit(ctx); err != nil {
		return nil, err
	}

	if ownerTwitchID != "" {
		s.auditUserRoleChange(ctx, rbacrpcserver.UserRoleAction{
			EntityTwitchID: ownerTwitchID,
			ActionFormat:   "Edit: Assigned company role of %s",
			Role:           params.OwnersNewRole,
			CompanyID:      params.CompanyId,
		})
	}

	s.auditUserRoleChange(ctx, rbacrpcserver.UserRoleAction{
		EntityTwitchID: params.TwitchIdToPromote,
		ActionFormat:   "Edit: Assigned company role of %s",
		Role:           "Owner",
		CompanyID:      params.CompanyId,
	})

	return &rbacrpc.Empty{}, nil
}

func (s *Server) auditUserRoleChange(ctx context.Context, a rbacrpcserver.UserRoleAction) {
	if a.CurrentUserTwitchID == "" {
		a.CurrentUserTwitchID = auth.GetTwitchID(ctx)
	}

	s.ActionHistories.InsertActionHistory(ctx, &actionhistories.ActionHistory{
		UserTwitchID: a.CurrentUserTwitchID,
		Action:       fmt.Sprintf(a.ActionFormat, a.Role),
		EntityType:   "UserRole",
		EntityID:     a.EntityTwitchID,
		CompanyID:    common.NewSQLNullString(a.CompanyID),
	})
}
