package rbacrpcserver

import (
	"fmt"

	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/clients/dart"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/backend/actionhistories"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/models/permissions"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"

	"context"

	"github.com/twitchtv/twirp"
)

// AddUserToCompany creates a new Membership of a user into an organization
func (s *Server) AddUserToCompany(ctx context.Context, params *rbacrpc.AddUserToCompanyRequest) (*rbacrpc.Empty, error) {
	if err := s.validateAddUserToCompanyParams(ctx, params); err != nil {
		return nil, err
	}
	memb := memberships.Membership{
		CompanyID: params.CompanyId,
		TwitchID:  params.User.TwitchId,
		Role:      params.Role,
		FirstName: params.User.FirstName,
		LastName:  params.User.LastName,
		DevTitle:  params.User.Title,
		DevEmail:  params.User.Email,
	}

	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	// Ensure company exists
	company, err := s.Backend.SelectCompany(ctx, memb.CompanyID)
	if err != nil {
		return nil, twirp.InvalidArgumentError("company_id", "company not found")
	}

	// Ensure user exists in the users-service (vaidate TwitchID)
	valid, err := s.Users.IsValidTwitchID(ctx, memb.TwitchID)
	if err != nil {
		return nil, err
	}
	if !valid {
		return nil, twirp.InvalidArgumentError("twitch_id", "is invalid or user with this numeric id does not exist")
	}

	userProps, errUser := s.Users.GetUserByID(ctx, memb.TwitchID)
	if errUser != nil || userProps == nil {
		return nil, errUser
	}

	// verify the user's email is verified
	if !(*userProps.EmailVerified) {
		return nil, twirp.InvalidArgumentError("twitch_id", "is invalid due to not having a verified email address")
	}

	// verify user has 2fa enabled
	twoFactorEnabled, err2fa := s.Passport.GetTwoFactorEnabled(ctx, memb.TwitchID)
	if err2fa != nil {
		return nil, err2fa
	}
	if !twoFactorEnabled {
		return nil, twirp.InvalidArgumentError("twitch_id", "is invalid due to not having a 2FA enabled")
	}

	// Ensure permissions
	if !auth.IsWhitelistAdmin(ctx) {
		addUserPermission := permissions.AddUser
		if params.Role == billingManagerRole {
			addUserPermission = permissions.AddUserBillingManager
		}

		// Get requesting user role in company
		requestingMemb, err := s.Memberships.GetMembership(ctx, params.CompanyId, params.RequestingTwitchId)
		if errorutil.IsErrNoRows(err) {
			return nil, twirp.NewError(twirp.PermissionDenied, "membership for current user (requesting_twitch_id, company_id) not found")
		}
		if err != nil {
			return nil, err
		}

		if !doesRoleHavePermission(requestingMemb.Role, addUserPermission) {
			return nil, twirp.NewError(twirp.PermissionDenied, "cannot add user to company")
		}

		// If the current users role is Manager, respond with permission denied when the params the role is owner or admin
		if requestingMemb.Role == managerRole {
			if params.Role == adminRole {
				return nil, twirp.NewError(twirp.PermissionDenied, "cannot add member with role "+params.Role)
			}
		}
	}

	// Ensure billing_manager is possible if assigned
	if params.Role == billingManagerRole {
		if err := s.validateCanBeBillingManager(ctx, memb.TwitchID); err != nil {
			return nil, err
		}
	}

	// Ensure the user is not already a member in the same org (prevent duplicates) or another org (prevent multi-org)
	otherMembs, _, err := s.Memberships.ListMemberships(ctx, memberships.ListMembershipsParams{
		TwitchID: memb.TwitchID,
		Limit:    100,
	})
	if err != nil {
		return nil, err
	}

	for _, membs := range otherMembs {
		if membs.CompanyID == params.CompanyId {
			return nil, twirp.NewError(twirp.AlreadyExists, "Membership already exists")
		}
	}

	// Create Membership
	err = s.Memberships.InsertMembership(ctx, &memb)
	if err != nil {
		return nil, err
	}

	if err := s.Backend.Commit(ctx); err != nil {
		return nil, err
	}

	if err := s.updateAllowedChattersByCompanyID(ctx, params.CompanyId); err != nil {
		logx.Error(ctx, errx.Wrap(err, "Failed to updateAllowedChattersByCompanyID, companyID: "+params.CompanyId))
	}

	s.auditUserRoleChange(ctx, UserRoleAction{
		CurrentUserTwitchID: params.RequestingTwitchId,
		EntityTwitchID:      memb.TwitchID,
		ActionFormat:        "Edit: Assigned company role of %s",
		Role:                memb.Role,
		CompanyID:           memb.CompanyID,
	})

	asyncWithLogger(ctx, func(ctx context.Context, lf logFunc) error {
		evsKey := company.Id
		devEmail := memb.DevEmail
		purpose := company.CompanyName

		emailTraceId, emailError := s.Dart.SendDeveloperCompanyUserActionAdded(ctx, memb.TwitchID, company.CompanyName) // if email was sent via dart we can track the status of that message with this

		s.auditEmailAction(ctx, DartEmailAction{
			UserTwitchID:   params.RequestingTwitchId,
			EntityTwitchID: memb.TwitchID,
			ActionFormat:   fmt.Sprintf("Email: Sent type %s with traceid of %s", dart.DeveloperCompanyUserActionAdded, emailTraceId),
			CompanyID:      company.Id,
		})

		if err := s.EVS.SendVerificationEmail(ctx, evsKey, devEmail, purpose); err != nil {
			lf(ctx, errx.Wrap(err, "Failed to send user a verification email."))
			// won't return here because we'd still like to send the notification to default twitch email
		}

		return emailError
	}, logx.Fields{
		"twitch_id":  memb.TwitchID,
		"company_id": company.Id,
	})

	return &rbacrpc.Empty{}, nil
}

func (s *Server) validateAddUserToCompanyParams(ctx context.Context, params *rbacrpc.AddUserToCompanyRequest) error {
	if auth.IsWhitelistAdmin(ctx) {
		params.RequestingTwitchId = auth.GetTwitchID(ctx)
	}

	if params.User == nil {
		return twirp.RequiredArgumentError("user")
	}
	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"twitch_id", params.User.TwitchId},
		{"company_id", params.CompanyId},
		{"role", params.Role},
		{"first_name", params.User.FirstName},
		{"last_name", params.User.LastName},
		{"dev_title", params.User.Title},
		{"requesting_twitch_id", params.RequestingTwitchId},
	}); err != nil {
		return err
	}

	if err := errorutil.ValidateUUID("company_id", params.CompanyId); err != nil {
		return err
	}

	// Validate role, and make sure the param is properly formatted to simplify equality checks
	formattedRole, err := SanitizeRole("role", params.Role)
	if err != nil {
		return err
	}
	params.Role = formattedRole

	if params.Role == "Owner" {
		return twirp.NewError(twirp.PermissionDenied, "User can not be set to Owner")
	}

	return nil
}

type UserRoleAction struct {
	CurrentUserTwitchID string // twitchID of the main actor of the update
	EntityTwitchID      string // twitchID of the membership that is being updated
	ActionFormat        string // description with "%s"; 1st %s is Role, 2nd %s is CompanyName
	Role                string
	CompanyID           string
}

func (s *Server) auditUserRoleChange(ctx context.Context, a UserRoleAction) {
	if a.CurrentUserTwitchID == "" {
		a.CurrentUserTwitchID = auth.GetTwitchID(ctx)
	}

	s.ActionHistories.InsertActionHistory(ctx, &actionhistories.ActionHistory{
		UserTwitchID: a.CurrentUserTwitchID,
		Action:       fmt.Sprintf(a.ActionFormat, a.Role),
		EntityType:   "UserRole",
		EntityID:     a.EntityTwitchID,
		CompanyID:    common.NewSQLNullString(a.CompanyID),
	})
}
