package rbacrpcserver

import (
	"context"
	"testing"
	"time"

	"github.com/stretchr/testify/require"
)

func TestAsync(t *testing.T) {
	ch := make(chan interface{}, 1)

	testFunc := func(ctx context.Context) error {
		requireContextNotDone(t, ctx)
		ch <- struct{}{}
		return nil
	}

	async(context.Background(), testFunc)
	select {
	case <-ch:
	case <-time.After(100 * time.Millisecond):
		t.Error("channel never got message")
	}
}

func TestAsyncContextCancelled(t *testing.T) {
	ch := make(chan interface{}, 1)

	testFunc := func(ctx context.Context) error {
		requireContextNotDone(t, ctx)
		ch <- struct{}{}
		return nil
	}

	ctx, cancel := context.WithCancel(context.Background())
	cancel()

	async(ctx, testFunc)
	select {
	case <-ch:
	case <-time.After(100 * time.Millisecond):
		t.Error("channel never got message")
	}
}

func TestAsyncWithLogger(t *testing.T) {
	ch := make(chan interface{}, 1)

	testFunc := func(ctx context.Context, lf logFunc) error {
		requireContextNotDone(t, ctx)
		require.NotNil(t, lf)
		ch <- struct{}{}
		return nil
	}

	asyncWithLogger(context.Background(), testFunc)
	select {
	case <-ch:
	case <-time.After(100 * time.Millisecond):
		t.Error("channel never got message")
	}
}

func TestAsyncLoggerContextCancelled(t *testing.T) {
	ch := make(chan interface{}, 1)

	testFunc := func(ctx context.Context, lf logFunc) error {
		requireContextNotDone(t, ctx)
		require.NotNil(t, lf)
		ch <- struct{}{}
		return nil
	}

	ctx, cancel := context.WithCancel(context.Background())
	cancel()

	asyncWithLogger(ctx, testFunc)
	select {
	case <-ch:
	case <-time.After(100 * time.Millisecond):
		t.Error("channel never got message")
	}
}

func requireContextNotDone(t *testing.T, ctx context.Context) {
	select {
	case <-ctx.Done():
		require.Fail(t, "ctx is done")
	default:
	}
}
