package rbacrpcserver

import (
	"context"
	"database/sql"
	"testing"

	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/backend/companyinvites/companyinvitesfakes"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships/membershipsfakes"
	"code.justin.tv/devrel/devsite-rbac/clients/dart/dartfakes"
	"code.justin.tv/devrel/devsite-rbac/clients/passport/passportfakes"
	"code.justin.tv/devrel/devsite-rbac/clients/users"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/stretchr/testify/require"
)

func NewCompanyInviteTestServer(t *testing.T, companyName string, inviterRole string, fakeCompanyInvite rbacrpc.CompanyInvite) *Server {
	server, rbacBackend := NewTestServer()
	fakeMemberships := &membershipsfakes.FakeMemberships{}
	server.Memberships = fakeMemberships
	fakeInvites := &companyinvitesfakes.FakeCompanyInvites{}
	server.CompanyInvites = fakeInvites
	usersFake := &users.NoopClient{}
	server.Users = usersFake
	dartFake := &dartfakes.FakeDart{}
	server.Dart = dartFake
	fakePassport := &passportfakes.FakeClient{}
	server.Passport = fakePassport
	request := validUserCompanyInviteRequestTemp()
	rbacBackend.BeginReturns(ctxBck)
	rbacBackend.SelectCompanyInviteReturns(nil, sql.ErrNoRows)
	fakeMemberships.ListMembershipsReturns([]memberships.Membership{{}}, 0, nil)
	fakePassport.GetTwoFactorEnabledReturns(true, nil)

	resp, err := server.IsUserValidForCompanyInvite(ctxBck, &request)

	require.NoError(t, err)
	require.NotNil(t, resp)

	rbacBackend.SelectCompanyReturns(&rbacrpc.Company{CompanyName: companyName}, nil)
	isValidInvitee, _ := server.Users.IsValidTwitchID(ctxBck, fakeCompanyInvite.InviteeTwitchId)
	isValidInviter, _ := server.Users.IsValidTwitchID(ctxBck, fakeCompanyInvite.InviterTwitchId)
	fakeInvites.InsertCompanyInviteReturns(&fakeCompanyInvite, nil)
	require.True(t, isValidInvitee)
	require.True(t, isValidInviter)

	fakeMemberships.GetMembershipReturnsOnCall(0, memberships.Membership{
		CompanyID: fakeCompanyInvite.CompanyId,
		Role:      inviterRole,
		TwitchID:  fakeCompanyInvite.InviterTwitchId,
	}, nil)
	fakeMemberships.GetMembershipReturnsOnCall(1, memberships.Membership{}, sql.ErrNoRows)
	fakeMemberships.GetMembershipReturnsOnCall(2, memberships.Membership{
		CompanyID: fakeCompanyInvite.CompanyId,
		Role:      inviterRole,
		TwitchID:  fakeCompanyInvite.InviterTwitchId,
	}, nil)

	return server
}

func TestCreateCompanyInvite_Success(t *testing.T) {
	ctxBck := context.Background()
	companyID := common.NewUUID()
	inviterRole := "Administrator"
	companyName := "Twitch Gaming"
	fakeCompanyInvite := &rbacrpc.CompanyInvite{Id: common.NewUUID(),
		CompanyId:       companyID,
		InviteeTwitchId: "123",
		InviterTwitchId: "789",
		Role:            "Developer",
		CreatedAt:       common.TimeNowStr()}

	server := NewCompanyInviteTestServer(t, companyName, inviterRole, *fakeCompanyInvite)

	resp, err := server.CreateCompanyInvite(ctxBck,
		&rbacrpc.CreateCompanyInviteRequest{
			InviteeTwitchId: fakeCompanyInvite.InviteeTwitchId,
			InviterTwitchId: fakeCompanyInvite.InviterTwitchId,
			CompanyId:       fakeCompanyInvite.CompanyId,
			Role:            fakeCompanyInvite.Role,
		})

	require.NoError(t, err)
	require.NotNil(t, resp)
	require.Equal(t, fakeCompanyInvite.Id, resp.Id)
	require.Equal(t, fakeCompanyInvite.CompanyId, resp.CompanyId)
	require.Equal(t, fakeCompanyInvite.InviteeTwitchId, resp.InviteeTwitchId)
	require.Equal(t, fakeCompanyInvite.InviterTwitchId, resp.InviterTwitchId)
	require.Equal(t, fakeCompanyInvite.Role, resp.Role)
	require.Equal(t, fakeCompanyInvite.CreatedAt, resp.CreatedAt)
}

func TestCreateCompanyInvite_InviterIsDeveloper_PermissionDenied(t *testing.T) {
	ctxBck := context.Background()
	companyID := common.NewUUID()
	inviterRole := "Developer"
	companyName := "Twitch Gaming"
	fakeCompanyInvite := &rbacrpc.CompanyInvite{Id: common.NewUUID(),
		CompanyId:       companyID,
		InviteeTwitchId: "123",
		InviterTwitchId: "789",
		Role:            "Developer",
		CreatedAt:       common.TimeNowStr()}

	server := NewCompanyInviteTestServer(t, companyName, inviterRole, *fakeCompanyInvite)

	resp, err := server.CreateCompanyInvite(ctxBck,
		&rbacrpc.CreateCompanyInviteRequest{
			InviteeTwitchId: fakeCompanyInvite.InviteeTwitchId,
			InviterTwitchId: fakeCompanyInvite.InviterTwitchId,
			CompanyId:       fakeCompanyInvite.CompanyId,
			Role:            fakeCompanyInvite.Role,
		})

	require.EqualError(t, err, "twirp error permission_denied: cannot invite user to company")
	require.Nil(t, resp)
}

func TestCreateCompanyInvite_MissingInviterRole_PermissionDenied(t *testing.T) {
	ctxBck := context.Background()
	companyID := common.NewUUID()
	inviterRole := ""
	companyName := "Twitch Gaming"
	fakeCompanyInvite := &rbacrpc.CompanyInvite{Id: common.NewUUID(),
		CompanyId:       companyID,
		InviteeTwitchId: "123",
		InviterTwitchId: "789",
		Role:            "Developer",
		CreatedAt:       common.TimeNowStr()}

	server := NewCompanyInviteTestServer(t, companyName, inviterRole, *fakeCompanyInvite)

	resp, err := server.CreateCompanyInvite(ctxBck,
		&rbacrpc.CreateCompanyInviteRequest{
			InviteeTwitchId: fakeCompanyInvite.InviteeTwitchId,
			InviterTwitchId: fakeCompanyInvite.InviterTwitchId,
			CompanyId:       fakeCompanyInvite.CompanyId,
			Role:            fakeCompanyInvite.Role,
		})

	require.EqualError(t, err, "twirp error permission_denied: cannot invite user to company")
	require.Nil(t, resp)
}

func TestCreateCompanyInvite_MissingInviter(t *testing.T) {
	ctxBck := context.Background()
	companyID := common.NewUUID()
	inviterRole := ""
	companyName := "Twitch Gaming"
	fakeCompanyInvite := &rbacrpc.CompanyInvite{Id: common.NewUUID(),
		CompanyId:       companyID,
		InviteeTwitchId: "123",
		InviterTwitchId: "",
		Role:            "Developer",
		CreatedAt:       common.TimeNowStr()}

	server := NewCompanyInviteTestServer(t, companyName, inviterRole, *fakeCompanyInvite)

	resp, err := server.CreateCompanyInvite(ctxBck,
		&rbacrpc.CreateCompanyInviteRequest{
			InviteeTwitchId: fakeCompanyInvite.InviteeTwitchId,
			InviterTwitchId: fakeCompanyInvite.InviterTwitchId,
			CompanyId:       fakeCompanyInvite.CompanyId,
			Role:            fakeCompanyInvite.Role,
		})

	require.EqualError(t, err, "twirp error invalid_argument: inviter_twitch_id is required")
	require.Nil(t, resp)
}

func TestCreateCompanyInvite_MissingInvitee(t *testing.T) {
	ctxBck := context.Background()
	companyID := common.NewUUID()
	inviterRole := "Administrator"
	companyName := "Twitch Gaming"
	fakeCompanyInvite := &rbacrpc.CompanyInvite{Id: common.NewUUID(),
		CompanyId:       companyID,
		InviteeTwitchId: "",
		InviterTwitchId: "789",
		Role:            "",
		CreatedAt:       common.TimeNowStr()}

	server := NewCompanyInviteTestServer(t, companyName, inviterRole, *fakeCompanyInvite)

	resp, err := server.CreateCompanyInvite(ctxBck,
		&rbacrpc.CreateCompanyInviteRequest{
			InviteeTwitchId: fakeCompanyInvite.InviteeTwitchId,
			InviterTwitchId: fakeCompanyInvite.InviterTwitchId,
			CompanyId:       fakeCompanyInvite.CompanyId,
			Role:            fakeCompanyInvite.Role,
		})

	require.EqualError(t, err, "twirp error invalid_argument: invitee_twitch_id is required")
	require.Nil(t, resp)
}
