package rbacrpcserver

import (
	"context"
	"fmt"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"

	"github.com/twitchtv/twirp"

	"code.justin.tv/devrel/devsite-rbac/backend/actionhistories"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/clients/dart"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

func (s *Server) DeleteGameApplication(ctx context.Context, params *rbacrpc.DeleteGameApplicationRequest) (*rbacrpc.Empty, error) {
	gameApp, err := s.Backend.SelectGameApplication(ctx, params.GetId())
	if errorutil.IsErrNoRows(err) {
		return nil, twirp.InvalidArgumentError("id", "game application not found")
	}
	if err != nil {
		return nil, err
	}

	err = s.Backend.DeleteGameApplication(ctx, params)
	if err != nil {
		return nil, err
	}

	s.sendGameApplicationRejectionEmail(ctx, params, gameApp)

	s.auditDeleteGameApplication(ctx, gameApp)
	return &rbacrpc.Empty{}, nil
}

func (s *Server) sendGameApplicationRejectionEmail(ctx context.Context, params *rbacrpc.DeleteGameApplicationRequest, gameApp *rbacrpc.GameApplication) {
	if params.SkipEmail {
		return
	}

	async(ctx, func(ctx context.Context) error {
		// Get game name
		gameID := fmt.Sprintf("%d", gameApp.GetGameId())
		games, err := s.Discovery.GetAll(ctx, []string{})
		if err != nil {
			return errx.Wrap(err, "Failed to get game name for game rejection email.")
		}
		gameName := ""
		game := games[gameID]
		if game != nil {
			gameName = game.Name
		}

		// Get user to send the email
		owner, err := s.getCompanyOwner(ctx, gameApp.CompanyId)
		if err != nil {
			return errx.Wrap(err, "Failed to get user to send game rejection email to.")
		}

		emailTraceId, emailError := s.Dart.SendDeveloperCompanyGameActionDenied(ctx, owner.TwitchID, gameName) // if email was sent via dart we can track the status of that message with this

		s.auditEmailAction(ctx, DartEmailAction{
			UserTwitchID:   owner.TwitchID,
			EntityTwitchID: gameApp.CompanyId,
			ActionFormat:   fmt.Sprintf("Email: Sent type %s with traceid of %s", dart.DeveloperCompanyGameActionDenied, emailTraceId),
			CompanyID:      gameApp.CompanyId,
		})

		return emailError
	}, logx.Fields{
		"gameApplicationID": params.Id,
	})

}

func (s *Server) auditDeleteGameApplication(ctx context.Context, gameApp *rbacrpc.GameApplication) {
	gameData, err := s.getGameByGameID(ctx, gameApp.GameId)
	if err != nil {
		logx.Error(ctx, errx.Wrap(err, "failed to get game for game activity log"))
	}

	s.ActionHistories.InsertActionHistory(ctx, &actionhistories.ActionHistory{
		UserTwitchID: auth.GetTwitchID(ctx),
		Action:       "Game application denied",
		EntityType:   "Game",
		EntityID:     common.Identifier(gameData.Name),
		CompanyID:    common.NewSQLNullString(gameApp.CompanyId),
	})
}

func (s *Server) getCompanyOwner(ctx context.Context, companyID string) (*memberships.Membership, error) {
	list, _, err := s.Memberships.ListMemberships(ctx, memberships.ListMembershipsParams{
		CompanyID: companyID,
		Role:      "Owner",
		Limit:     1,
	})
	if err != nil {
		return nil, err
	}
	if len(list) == 0 {
		return nil, errx.New("company has no owner user")
	}
	return &list[0], nil
}
