package rbacrpcserver

import (
	"context"

	"github.com/twitchtv/twirp"

	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/models/permissions"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

func (s *Server) GetExtensionBillingManager(ctx context.Context, params *rbacrpc.GetExtensionBillingManagerRequest) (*rbacrpc.GetExtensionBillingManagerResponse, error) {
	if err := s.validateGetBillingManagerParams(ctx, params); err != nil {
		return nil, err
	}

	// Check permissions
	if !auth.IsWhitelistAdmin(ctx) {
		validResp, err := s.ValidateByTwitchID(ctx, &rbacrpc.ValidateQuery{
			UserId:       params.RequestingTwitchId,
			ResourceId:   params.ExtensionClientId,
			Permission:   permissions.ManageExtensions,
			ResourceType: permissions.Extension,
		})
		if err != nil {
			return nil, err
		}
		if !validResp.Valid {
			return nil, twirp.NewError(twirp.PermissionDenied, "not allowed to manage extensions")
		}
	}

	extensionBillingManager, err := s.Backend.GetExtensionBillingManager(ctx, params.ExtensionClientId)
	if errorutil.IsErrNoRows(err) {
		return nil, twirp.NewError(twirp.NotFound, "No rows were returned")
	}
	if err != nil {
		return nil, err
	}

	// check if billing manager account is not deleted
	user, err := s.Users.GetUserByID(ctx, extensionBillingManager.BillingManagerTwitchID)
	if err != nil {
		return nil, err
	}

	isBillingManagerAccountDeleted := (user.DeletedOn != nil)

	return &rbacrpc.GetExtensionBillingManagerResponse{
		ExtensionBillingManager:        extensionBillingManager.ToRPC(),
		IsBillingManagerAccountDeleted: isBillingManagerAccountDeleted,
	}, nil
}

func (s *Server) validateGetBillingManagerParams(ctx context.Context, params *rbacrpc.GetExtensionBillingManagerRequest) error {
	if auth.IsWhitelistAdmin(ctx) {
		params.RequestingTwitchId = auth.GetTwitchID(ctx)
	}

	return errorutil.ValidateRequiredArgs(errorutil.Args{
		{"extension_client_id", params.ExtensionClientId},
		{"requesting_twitch_id", params.RequestingTwitchId},
	})
}
