package rbacrpcserver

import (
	"database/sql"
	"testing"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships/membershipsfakes"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/stretchr/testify/require"
)

func TestGetMembership_Success(t *testing.T) {
	server, _ := NewTestServer()
	fakeMemberships := &membershipsfakes.FakeMemberships{}

	uuid := common.NewUUID()

	fakeMemberships.GetMembershipReturns(memberships.Membership{
		CompanyID: uuid,
		Role:      "Owner",
		FirstName: "McDudeCool",
	}, nil)
	server.Memberships = fakeMemberships

	resp, err := server.GetMembership(ctxBck, &rbacrpc.GetMembershipRequest{
		CompanyId:          uuid,
		TwitchId:           "123",
		RequestingTwitchId: "123",
	})
	require.NoError(t, err)

	require.Equal(t, uuid, resp.CompanyId)
	require.Equal(t, "Owner", resp.Role)
	require.Equal(t, "McDudeCool", resp.FirstName)
}

func TestGetMembership_RequiredArgs(t *testing.T) {
	server, _ := NewTestServer()

	_, err := server.GetMembership(ctxBck, &rbacrpc.GetMembershipRequest{
		CompanyId: common.NewUUID(),
		TwitchId:  "123",
	})
	require.EqualError(t, err, "twirp error invalid_argument: requesting_twitch_id is required")

	_, err = server.GetMembership(ctxBck, &rbacrpc.GetMembershipRequest{
		CompanyId:          common.NewUUID(),
		RequestingTwitchId: "123",
	})
	require.EqualError(t, err, "twirp error invalid_argument: twitch_id is required")

	_, err = server.GetMembership(ctxBck, &rbacrpc.GetMembershipRequest{
		TwitchId:           "123",
		RequestingTwitchId: "123",
	})
	require.EqualError(t, err, "twirp error invalid_argument: company_id is required")
}

func TestGetMembership_NotFound(t *testing.T) {
	server, _ := NewTestServer()
	fakeMemberships := &membershipsfakes.FakeMemberships{}
	fakeMemberships.GetMembershipReturns(memberships.Membership{}, errx.Wrap(sql.ErrNoRows, "not found"))
	server.Memberships = fakeMemberships

	_, err := server.GetMembership(ctxBck, &rbacrpc.GetMembershipRequest{
		CompanyId:          common.NewUUID(),
		TwitchId:           "123",
		RequestingTwitchId: "123",
	})
	require.EqualError(t, err, "twirp error not_found: Membership not found")
}
