package rbacrpcserver

import (
	"context"
	"fmt"

	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/twitchtv/twirp"
)

const (
	defaultLimit = 50
	maxLimit     = 1000
)

// GetPaginatedShadowAccounts returns paginated shadow accounts twitch IDs and next query offset
func (s *Server) GetPaginatedShadowAccounts(ctx context.Context, params *rbacrpc.GetPaginatedShadowAccountsRequest) (*rbacrpc.GetPaginatedShadowAccountsResponse, error) {
	listMembershipParams := memberships.ListMembershipsParams{
		Role:    shadowAccountRole,
		OrderBy: "twitch_id",
	}

	if params.Limit > maxLimit {
		return nil, twirp.NewError(twirp.InvalidArgument, fmt.Sprintf("The limit exceeds the maximum number: %d", maxLimit))
	}

	if params.Limit != 0 {
		listMembershipParams.Limit = params.Limit
	} else {
		// otherwise default limit will be 20
		listMembershipParams.Limit = defaultLimit
	}

	if params.Offset != 0 {
		listMembershipParams.Offset = params.Offset
	}

	memberships, _, err := s.Memberships.ListMemberships(ctx, listMembershipParams)
	if err != nil {
		return nil, err
	}

	twitchIDs := []string{}
	for _, membership := range memberships {
		twitchIDs = append(twitchIDs, membership.TwitchID)
	}

	return &rbacrpc.GetPaginatedShadowAccountsResponse{
		NextOffset: params.Offset + uint64(len(memberships)),
		TwitchIds:  twitchIDs,
	}, nil
}
